#charset "utf-8"

/* 
 *   Czech translation is copyright 2010 Tomas Blaha.
 *
 *   Based on English language module by Michael J. Roberts with permisson.
 *   Original copyright below:
 *
 *   Copyright (c) 2000, 2006 Michael J. Roberts.  All Rights Reserved. 
 *   
 *   TADS 3 Library: Instructions for new players
 *   
 *   This module defines the INSTRUCTIONS command, which provides the
 *   player with an overview of how to play IF games in general.  These
 *   instructions are especially designed as an introduction to IF for
 *   inexperienced players.  The instructions given here are meant to be
 *   general enough to apply to most games that follow the common IF
 *   conventions. 
 *   
 *   This module defines the English version of the instructions.
 *   
 *   In most cases, each author should customize these general-purpose
 *   instructions at least a little for the specific game.  We provide a
 *   few hooks for some specific parameter-driven customizations that don't
 *   require modifying the original text in this file.  Authors should also
 *   feel free to make more extensive customizations as needed to address
 *   areas where the game diverges from the common conventions described
 *   here.
 *   
 *   One of the most important things you should do to customize these
 *   instructions for your game is to add a list of any special verbs or
 *   command phrasings that your game uses.  Of course, you might think
 *   you'll be spoiling part of the challenge for the player if you do
 *   this; you might worry that you'll give away a puzzle if you don't keep
 *   a certain verb secret.  Be warned, though, that many players - maybe
 *   even most - don't think "guess the verb" puzzles are good challenges;
 *   a lot of players feel that puzzles that hinge on finding the right
 *   verb or phrasing are simply bad design that make a game less
 *   enjoyable.  You should think carefully about exactly why you don't
 *   want to disclose a particular verb in the instructions.  If you want
 *   to withhold a verb because the entire puzzle is to figure out what
 *   command to use, then you have created a classic guess-the-verb puzzle,
 *   and most everyone in the IF community will feel this is simply a bad
 *   puzzle that you should omit from your game.  If you want to withhold a
 *   verb because it's too suggestive of a particular solution, then you
 *   should at least make sure that a more common verb - one that you are
 *   willing to disclose in the instructions, and one that will make as
 *   much sense to players as your secret verb - can achieve the same
 *   result.  You don't have to disclose every *accepted* verb or phrasing
 *   - as long as you disclose every *required* verb *and* phrasing, you
 *   will have a defense against accusations of using guess-the-verb
 *   puzzles.
 *   
 *   You might also want to mention the "cruelty" level of the game, so
 *   that players will know how frequently they should save the game.  It's
 *   helpful to point out whether or not it's possible for the player
 *   character to be killed; whether it's possible to get into situations
 *   where the game becomes "unwinnable"; and, if the game can become
 *   unwinnable, whether or not this will become immediately clear.  The
 *   kindest games never kill the PC and are always winnable, no matter
 *   what actions the player takes; it's never necessary to save these
 *   games except to suspend a session for later resumption.  The cruelest
 *   games kill the PC without warning (although if they offer an UNDO
 *   command from a "death" prompt, then even this doesn't constitute true
 *   cruelty), and can become unwinnable in ways that aren't readily and
 *   immediately apparent to the player, which means that the player could
 *   proceed for quite some time (and thus invest substantial effort) after
 *   the game is already effectively lost.  Note that unwinnable situations
 *   can often be very subtle, and might not even be intended by the
 *   author; for example, if the player needs a candle to perform an
 *   exorcism at some point, but the candle can also be used for
 *   illumination in dark areas, the player could make the game unwinnable
 *   simply by using up the candle early on while exploring some dark
 *   tunnels, and might not discover the problem until much further into
 *   the game.  
 */

#include "adv3.h"
#include "cs_cz.h"

/*
 *   The INSTRUCTIONS command.  Make this a "system" action, because it's
 *   a meta-action outside of the story.  System actions don't consume any
 *   game time.  
 */
DefineSystemAction(Instructions)
    /*
     *   This property tells us how complete the verb list is.  By default,
     *   we'll assume that the instructions fail to disclose every required
     *   verb in the game, because the generic set we use here doesn't even
     *   try to anticipate the special verbs that most games include.  If
     *   you provide your own list of game-specific verbs, and your custom
     *   list (taken together with the generic list) discloses every verb
     *   required to complete the game, you should set this property to
     *   true; if you set this to true, the instructions will assure the
     *   player that they will not need to think of any verbs besides the
     *   ones listed in the instructions.  Authors are strongly encouraged
     *   to disclose a list of verbs that is sufficient by itself to
     *   complete the game, and to set this property to true once they've
     *   done so.  
     */
    allRequiredVerbsDisclosed = nil

    /* 
     *   A list of custom verbs.  Each game should set this to a list of
     *   single-quoted strings; each string gives an example of a verb to
     *   display in the list of sample verbs.  Something like this:
     *   
     *   customVerbs = ['brush my teeth', 'pick the lock'] 
     */
    customVerbs = []

    /* 
     *   Verbs relating specifically to character interaction.  This is in
     *   the same format as customVerbs, and has essentially the same
     *   purpose; however, we call these out separately to allow each game
     *   not only to supplement the default list we provide but to replace
     *   our default list.  This is desirable for conversation-related
     *   commands in particular because some games will not use the
     *   ASK/TELL conversation system at all and will thus want to remove
     *   any mention of the standard set of verbs.  
     */
    conversationVerbs =
    [
        'ZEPTEJ SE MÁGA NA HŮLKU',
        'ŘEKNI MÁGOVI O ZAPRÁŠENÉ KNIZE',
        'POŽÁDEJ MÁGA O LEKTVAR',
        'UKAŽ MÁGOVI SVITEK',
        'PODEJ MÁGOVI LOPATU',
        'ANO (nebo NE)'
    ]

    /* conversation verb abbreviations */
    conversationAbbr = "\n\tZEPTEJ SE NA (téma) může být zkráceno na ZEP (téma)
        \n\tŘEKNI O (tématu) může být zadáno jako ŘEK (téma)"

    /*
     *   Truncation length. If the game's parser allows words to be
     *   abbreviated to some minimum number of letters, this should
     *   indicate the minimum length.  The English parser uses a truncation
     *   length of 6 letters by default.
     *   
     *   Set this to nil if the game doesn't allow truncation at all.  
     */
    truncationLength = 6

    /*
     *   This property should be set on a game-by-game basis to indicate
     *   the "cruelty level" of the game, which is a rough estimation of
     *   how likely it is that the player will encounter an unwinnable
     *   position in the game.
     *   
     *   Level 0 is "kind," which means that the player character can
     *   never be killed, and it's impossible to make the game unwinnable.
     *   When this setting is used, the instructions will reassure the
     *   player that saving is necessary only to suspend the session.
     *   
     *   Level 1 is "standard," which means that the player character can
     *   be killed, and/or that unwinnable positions are possible, but
     *   that there are no especially bad unwinnable situations.  When
     *   this setting is selected, we'll warn the player that they should
     *   save every so often.
     *   
     *   (An "especially bad" situation is one in which the game becomes
     *   unwinnable at some point, but this won't become apparent to the
     *   player until much later.  For example, suppose the first scene
     *   takes place in a location that can never be reached again after
     *   the first scene, and suppose that there's some object you can
     *   obtain in this scene.  This object will be required in the very
     *   last scene to win the game; if you don't have the object, you
     *   can't win.  This is an "especially bad" unwinnable situation: if
     *   you leave the first scene without getting the necessary object,
     *   the game is unwinnable from that point forward.  In order to win,
     *   you have to go back and play almost the whole game over again.
     *   Saved positions are almost useless in a case like this, since
     *   most of the saved positions will be after the fatal mistake; no
     *   matter how often you saved, you'll still have to go back and do
     *   everything over again from near the beginning.)
     *   
     *   Level 2 is "cruel," which means that the game can become
     *   unwinnable in especially bad ways, as described above.  If this
     *   level is selected, we'll warn the player more sternly to save
     *   frequently.
     *   
     *   We set this to 1 ("standard") by default, because even games that
     *   aren't intentionally designed to be cruel often have subtle
     *   situations where the game becomes unwinnable, because of things
     *   like the irreversible loss of an object, or an unrepeatable event
     *   sequence; it almost always takes extra design work to ensure that
     *   a game is always winnable.  
     */
    crueltyLevel = 1

    /*
     *   Does this game have any real-time features?  If so, set this to
     *   true.  By default, we'll explain that game time passes only in
     *   response to command input. 
     */
    isRealTime = nil

    /*
     *   Conversation system description.  Several different conversation
     *   systems have come into relatively widespread use, so there isn't
     *   any single convention that's generic enough that we can assume it
     *   holds for all games.  In deference to this variability, we
     *   provide this hook to make it easy to replace the instructions
     *   pertaining to the conversation system.  If the game uses the
     *   standard ASK/TELL system, it can leave this list unchanged; if
     *   the game uses a different system, it can replace this with its
     *   own instructions.
     *   
     *   We'll include information on the TALK TO command if there are any
     *   in-conversation state objects in the game; if not, we'll assume
     *   there's no need for this command.
     *   
     *   We'll mention the TOPICS command if there are any SuggestedTopic
     *   instances in the game; if not, then the game will never have
     *   anything to suggest, so the TOPICS command isn't needed.
     *   
     *   We'll include information on special topics if there are any
     *   SpecialTopic objects defined.  
     */
    conversationInstructions =
        "S ostatními postavami se můžeš bavit tak, že se jich ptáš na věci
        vyskytující se v příběhu nebo jim o nich říkáš. Mohl bys např. napsat
        ZEPTEJ SE MÁGA NA HŮLKU nebo ŘEKNI STRÁŽCI O ALARMU. Vždy bys měl
        používat vazbu ZEPTEJ SE NA nebo ŘEKNI O, protože příběh jiným formám
        nebude rozumět. Není potřeba vymýšlet žádné komplikované otázky jako <q>zeptej
        se strážného, jak se otevírá okno.</q> Ve většině případů je mnohem lepší se
        ptát na konkrétní objekty nebo postavy, na které jsi v příběhu narazil,
        než zkoušet dotazy na abstraktní témata jako SMYSL ŽIVOTA. Pokud by však
        v příběhu bylo něco, díky čemuž bys věřil, že se <i>máš</i> na určité
        abstraktní téma ptát, pak není důvod to nezkusit.

        \bKdyž budeš říkat nebo se ptát stejné osoby na více témat po sobě,
        můžeš si uštřit psaní zkrácením ZEPTEJ SE NA na ZEP a ŘEKNI O na ŘEK.
        Například jakmile už mluvíš s mágem, můžeš zkrátit ZEPTEJ SE MÁGA NA
        AMULET na pouhé ZEP AMULET. Tím je otázka směrována na stejnou postavu
        jako v předchozím ZEPTEJ SE NA či ŘEKNI O.

        <<firstObj(InConversationState, ObjInstances) != nil ?
          "\bPostavu ve hře pozdravíš napsáním MLUV S (postavou). Tím se pokusíš
          upoutat její pozornost a začít s ní konverzaci. MLUV S ale není
          povinné. Pokud chceš, můžeš vždy rovnou začít s příkazy ZEPTEJ SE NA
          nebo ŘEKNI O." : "">>

        <<firstObj(SpecialTopic, ObjInstances) != nil ?
          "\bPříběh může občas nabídnout zvláštní konverzační příkazy, jako např:

          \b\t(Mohl by ses omluvit nebo vysvětlit situaci.)

          \bPokud chceš, můžeš se řídit doporučením a zadat příkaz v nabízeném
          tvaru rozkazovacím způsobem. Obvykle stačí jen pár prvních slov:

          \b\t&gt;OMLUV SE
          \n\t&gt;VYSVĚTLI SITUACI

          \bPodobná doporučená sousloví fungují jen ve chvíli, kdy je hra
          nabídne, takže si nemusíš dělat starosti a zkoušet je jindy a jinde.
          Nejsou to nové příkazy k zapamatování, jen možnosti navíc v
          konkrétní situaci a příběh na ně vždy upozorní. Pokud se vyskytnou,
          nijak tě nelimitují v tom, co můžeš dělat. Vždy můžeš místo doporučení
          zadat jakýkoliv běžný příkaz." : "">>

        <<firstObj(SuggestedTopic, ObjInstances) != nil ?
          "\bNejseš-li si jistý, o čem by se dalo diskutovat, můžeš, kdykoliv se
          s někým bavíš, zadat příkaz TÉMATA. To ti nabídne seznam námětů, které
          může tvá postava použít v diskusi s jinou osobou. Příkaz TÉMATA obvykle
          neodhalí všechny možnosti, takže klidně zkus i další i přes to, že
          nejsou uvedeny." : "">>

        \bS ostatními postavami můžeš interagovat také pomocí fyzických objektů.
        Například bys mohl mít možnost něco dát jiné postavě příkazem DEJ PENÍZE
        ÚŘEDNÍKOVI nebo ukázat někomu objekt, třeba UKAŽ MODLU PROFESOROVI. S
        ostatními postavami můžeš také bojovat, třeba ZAÚTOČ NA TROLA MEČEM nebo
        HOĎ SEKERU PO TRPASLÍKOVI.

        \bV určitých situacích můžeš jinou postavu požádat, aby pro tebe něco
        udělala. Stačí napsat jméno postavy, za něj čárku a potom příkaz, který
        chceš, aby postava vykonala tím stejným způsobem, jako úkoluješ svojí
        postavu. Např.:

        \b\t&gt;ROBOTE, JDI NA SEVER

        \bMěj ale na paměti, že nemáš nikdy jistotu, že ostatní postavy budou
        vždy poslouchat tvá přání. Většina postav má svou vlastní vůli a nebudou
        automaticky dělat cokoliv, o co je požádáš. "

    /* execute the command */
    execSystemAction()
    {
        local origElapsedTime;

        /* 
         *   note the elapsed game time on the real-time clock before we
         *   start, so that we can reset the game time when we're done; we
         *   don't want the instructions display to consume any real game
         *   time 
         */
        origElapsedTime = realTimeManager.getElapsedTime();

        /* show the instructions */
        showInstructions();

        /* reset the real-time game clock */
        realTimeManager.setElapsedTime(origElapsedTime);
    }

#ifdef INSTRUCTIONS_MENU
    /*
     *   Show the instructions, using a menu-based table of contents.
     */
    showInstructions()
    {
        /* run the instructions menu */
        topInstructionsMenu.display();

        /* show an acknowledgment */
        "Zpátky v příběhu. ";
    }
    
#else /* INSTRUCTIONS_MENU */

    /*
     *   Show the instructions as a standard text display.  Give the user
     *   the option of turning on a SCRIPT file to capture the text.  
     */
    showInstructions()
    {
        local startedScript;

        /* presume we won't start a new script file */
        startedScript = nil;
        
        /* show the introductory message */
        "The story is about to show a full set of instructions,
        designed especially for people who aren&rsquo;t already familiar
        with interactive fiction.  The instructions are lengthy";

        /*
         *   Check to see if we're already scripting.  If we aren't, offer
         *   to save the instructions to a file. 
         */
        if (scriptStatus.scriptFile == nil)
        {
            local str;
            
            /* 
             *   they're not already logging; ask if they'd like to start
             *   doing so 
             */
            ", so you might want to capture them to a file (so that
            you can print them out, for example).  Would you like to
            proceed?
            \n(<<aHref('ano', 'A')>> znamená souhlas, popř. zadej
            <<aHref('zápis', 'ZÁPIS')>> k zápisu do souboru) &gt; ";

            /* ask for input */
            str = inputManager.getInputLine(nil, nil);

            /* if they want to capture them to a file, set up scripting */
            if (rexMatch('<nocase><space>*z(á(p(i(s?)?)?)?)?<space>*', str)
                == str.length())
            {
                /* try setting up a scripting file */
                ScriptAction.setUpScripting(nil);

                /* if that failed, don't proceed */
                if (scriptStatus.scriptFile == nil)
                    return;
                
                /* note that we've started a script file */
                startedScript = true;
            }
            else if (rexMatch('<nocase><space>*a.*', str) != str.length())
            {
                "Přerušeno. ";
                return;
            }
        }
        else
        {
            /* 
             *   they're already logging; just confirm that they want to
             *   see the instructions 
             */
            "; would you like to proceed?
            \n(A znamená souhlas) &gt; ";

            /* stop if they don't want to proceed */
            if (!yesOrNo())
            {
                "Přerušeno. ";
                return;
            }
        }

        /* make sure we have something for the next "\b" to skip from */
        "\ ";

        /* show each chapter in turn */
        showCommandsChapter();
        showAbbrevChapter();
        showTravelChapter();
        showObjectsChapter();
        showConversationChapter();
        showTimeChapter();
        showSaveRestoreChapter();
        showSpecialCmdChapter();
        showUnknownWordsChapter();
        showAmbiguousCmdChapter();
        showAdvancedCmdChapter();
        showTipsChapter();

        /* if we started a script file, close it */
        if (startedScript)
            ScriptOffAction.turnOffScripting(nil);
    }

#endif /* INSTRUCTIONS_MENU */

    /* Entering Commands chapter */
    showCommandsChapter()
    {
        "\b<b>Zadávání příkazů</b>\b
        Už sis asi všiml, že ovlivňuješ příběh psaním příkazů, kdykoliv uvidíš
        <q>výzvu,</q> která většinou vypadá nějak takhle:
        \b";

        gLibMessages.mainCommandPrompt(rmcCommand);

        "\bS touto znalostí tě pravděpodobně napadne jedna ze dvou věcí:
        <q>Skvělé, můžu napsat absolutně cokoliv chci, jen tak v češtině, a
        příběh splní všechna má přání,</q> nebo <q>Skvělé, teď budu muset
        pochopit další odporně složitý programovací jazyk pro ovládání počítačů.
        To si raději půjdu zahrát Miny.</q> Přesto ani jeden extrém není úplně
        pravdivý.

        \bVe skutečné hře si vystačíš s celkem malou množinou příkazů a ty jsou
        většinou v běžné češtině, takže toho není mnoho, co by ses měl učit nebo
        si musel pamatovat. Přestože příkazová řádka může vypadat strašidelně,
        nenech se vylekat &mdash; potřebuješ znát jen pár jednoduchých zásad.

        \bZaprvé téměř nikdy nebudeš potřebovat se odvolávat na něco, co není
        přímo zmíněné v příběhu; je to koneckonců příběh a ne hra na hádání, ve
        které bys měl přemýšlet, co všechno se pojí s některým náhodně vybraným
        objektem. Kupříkladu máš-li na sobě bundu, můžeš předpokládat, že má
        kapsy, knoflíky nebo zip &mdash; ale pokud se příběh o těchto detailech
        nikdy nezmiňuje, tak by ses jimi neměl zabývat.

        \bZadruhé nemusíš přemýšlet o každé myslitelné akci, kterou bys mohl
        provést. Pointa hry není v tom nechat tě hádat slovesa. Místo toho ti
        bude stačit používat relativně malý počet běžných základních akcí.
        Abys měl představu, co máme na mysli, tady jsou některé příkazy,
        jaké bys mohl použít:";
        
        "\b
        \n\t ROZHLÉDNI SE
        \n\t INVENTÁŘ
        \n\t JDI NA SEVER (VÝCHOD, JIHOZÁPAD,... NAHORU, DOLŮ, DOVNITŘ, VEN)
        \n\t ČEKEJ
        \n\t VEM KRABIČKU
        \n\t POLOŽ DISKETU
        \n\t KOUKNI SE NA DISKETU
        \n\t PŘEČTI KNIHU
        \n\t OTEVŘI KRABIČKU
        \n\t ZAVŘI KRABIČKU
        \n\t KOUKNI SE DO KRABIČKY
        \n\t KOUKNI SKRZ OKNO
        \n\t VLOŽ DISKETU DO KRABIČKY
        \n\t POLOŽ KRABIČKU NA STŮL
        \n\t NASAĎ SI ŠPIČATÝ KLOBOUK
        \n\t SUNDEJ SI KLOBOUK
        \n\t ZAPNI SVÍTILNU
        \n\t ZAPAL SIRKU
        \n\t ZAPAL SVÍČKU SIRKOU
        \n\t ZMÁČKNI TLAČÍTKO
        \n\t ZATÁHNI ZA PÁKU
        \n\t OTOČ KNOFLÍKEM
        \n\t NATOČ CIFERNÍK NA 11
        \n\t SNĚZ SUŠENKU
        \n\t VYPIJ MLÉKO
        \n\t HOĎ DORT NA KLAUNA
        \n\t ZAÚTOČ NA TROLA MEČEM
        \n\t ODEMKNI DVEŘE KLÍČEM
        \n\t ZAMKNI DVEŘE KLÍČEM
        \n\t VYLEZ PO ŽEBŘÍKU
        \n\t NASTUP DO AUTA
        \n\t SEDNI SI NA ŽIDLI
        \n\t STOUPNI SI NA STŮL
        \n\t STOUPNI SI DO STÍNU
        \n\t LEHNI SI NA POSTEL
        \n\t NAPIŠ AHOJ NA POČÍTAČI
        \n\t NAJDI HONZU V TELEFONNÍM SEZNAMU";

        /* show the conversation-related verbs */
        foreach (local cur in conversationVerbs)
            "\n\t <<cur>>";

        /* show the custom verbs */
        foreach (local cur in customVerbs)
            "\n\t <<cur>>";

        /* 
         *   if the list is exhaustive, say so; otherwise, mention that
         *   there might be some other verbs to find 
         */
        if (allRequiredVerbsDisclosed)
            "\bA to je vše &mdash; každé sloveso a každý tvar příkazu, který je
            nezbytný k dokončení příběhu je ukázán výše. Pokud se někdy zasekneš
            a budeš mít pocit, že příběh očekává, že vymyslíš něco zcela mimo
            běžnou rutinu, vzpomeň si na tohle: cokoliv, co máš udělat, dosáhneš
            jedním či více příkazy ukázanými výše. Nikdy neuvažuj o tom, že bys
            měl začít zkoušet spoustu náhodných příkazů, aby ses trefil do
            nějakého magického slovosledu, protože všechno, co potřebuješ, je
            uvedeno výše. ";
        else
            "\bVětšina sloves, která potřebuješ k dokončení příběhu, je ukázána
            výše. Může existovat pár dodatečných příkazů, které čas od času
            budeš také potřebovat, ale ty jsou utvořené stejným jednoduchým
            způsobem jako příkazy výše.";

        "\bNěkteré z těchto příkazů si zaslouží lépe vysvětlit. ROZHLÉDNI SE
        (které může být zkáceno na ROZ, nebo jen R) ukáže popis současné lokace.
        Tím si můžeš osvěžit paměť o tom, co tvou postavu obklopuje. INVENTÁŘ
        (nebo jen I) vypíše seznam všeho, co tvá postava nese s sebou. ČEKEJ
        (nebo Č) nechá uplynout trochu času v příběhu.";
    }

    /* Abbreviations chapter */
    showAbbrevChapter()
    {
        "\b<b>Zkracování příkazů</b>
        \bPár příkazů budeš pravděpodobně používat docela často, tak aby sis
        ušetřil psaní, můžeš je zkrátit:

        \b
        \n\t ROZHLÉDNI SE můžeš zkrátit na ROZ či pouhé R
        \n\t INVENTÁŘ zkrátíš na I
        \n\t JDI NA SEVER může být zapsáno jako SEVER nebo dokonce jen S (stejně
            tak V, Z, J, SV, JV, SZ, JZ, N pro NAHORU, D pro DOLŮ a DO
            pro DOVNITŘ)
        \n\t PODÍVEJ SE NA DISKETU může být zadáno jako PROZKOUMEJ DISKETU nebo
            jen P DISKETU
        \n\t ČEKEJ můžeš zkrátit na Č
        <<conversationAbbr>>

        \b<b>Několik dalších detailů</b>
        \bKdyž zadáváš příkazy, můžeš používat malá či velká písmena v libovolné
        kombinaci. ";

        if (truncationLength != nil)
        {
            "Kterékoliv slovo můžeš zkrátit na jeho první<<truncationLength >= 5
            ? 'ch' : ''>> <<spellInt(truncationLength, 4)>> písmen<<
            truncationLength < 5 ? 'a' : ''>>, ale pokud se rozhodneš
            nezkracovat, příběh bude dávat pozor na všechno, co napíšeš. To
            znamená, že např. můžeš zkrátit NEJNEOBHOSPODÁŘOVÁVATELNĚJŠÍMI na
            <<'NEJNEOBHOSPODÁŘOVÁVATELNĚJŠÍMI'.substr(1, truncationLength)>>
            nebo <<'NEJNEOBHOSPODÁŘOVÁVATELNĚJŠÍMI'.substr(1, truncationLength
            + 3)>>, ale ne na <<'NEJNEOBHOSPODÁŘOVÁVATELNĚJŠÍMI'.substr(1,
            truncationLength)>>SDF. ";
        }
    }

    /* Travel chapter */
    showTravelChapter()
    {
        "\b<b>Pohyb</b>
        \bV každém okamžiku se tvá postava nachází v <q>lokaci.</q> Příběh
        popíše lokaci, jakmile do ní poprvé vstoupíš a znovu kdykoliv zadáš
        ROZHLÉDNI SE. Každá lokace obvykle má krátký název, který je zobrazen
        před celým popisem. Název je užitečný při kreslení mapy a také pomáhá
        osvěžit paměť při hledání cesty po okolí.

        \bKaždá lokace je samostatnou místností nebo rozlehlou venkovní oblastí
        apod. (Někdy je jedna fyzická místnost tak rozlehlá, že je ve hře
        složena z několika lokací, ale to bývá jen zřídka.) Povětšinou je vstup
        do lokace tak specifický, že o něm uvažuješ jako o přesunu. Jakmile už
        je tvá postava v lokaci, obvykle vidí a dosáhne na všechno v ní
        obsažené, takže se nemusíš zabývat tím, kde přesně tvá postava v
        místnosti stojí. Čas od času můžeš zjistit, že něco je mimo dosah, třeba
        proto, že je to vysoko na polici nebo na opačné straně vodního příkopu. V
        těchto případech může někdy pomoci přesněji určit pozici své
        postavy, jako třeba na něčem stát (například STOUPNI SI NA STŮL).

        \bPřesun z jedné lokace do jiné se obvykle odehrává pomocí směrových
        příkazů: JDI NA SEVER, JDI SEVEROVÝCHODNĚ, JDI NAHORU atd. (Základní a
        vertikální směry můžeš každý zkrátit na jedno písmeno &mdash; S, J, V,
        Z, N, D &mdash; a úhlopříčné směry na dvě: SV, SZ, JV, JZ.) Příběh by
        měl vždy při popisu lokace říci všechny směry, kterými se lze vydat.
        Není tedy potřeba zkoušet směry, které nebyly zmíněny, zda náhodou
        někam nevedou.

        \bVe většině případů se lze spolehnout, že cesta přesně opačným směrem
        tě přivede zase zpátky do lokace, ze které jsi původně vyrazil. Některé
        cesty ale mohou mít zatáčky.

        \bPo většinou, když příběh popisuje dveře nebo nějaký průchod,
        nepotřebuješ je otevírat, abys jimi prošel. To příběh udělá za
        tebe. Jen pokud příběh konkrétně zmíní, že dveře blokují tvou cestu,
        tak si s nimi budeš muset sám poradit.";
    }

    /* Objects chapter */
    showObjectsChapter()
    {
        "\b<b>Manipulace s předměty</b>
        \bV příběhu pravděpodobně narazíš na objekty, které tvá postava může
        sebrat a různě s nimi manipulovat. Pokud chceš něco sebrat, napiš VEM a
        název objektu: VEM KNIHU. Pokud chceš něco z toho, co tvá postava nese,
        odložit, tak to POLOŽ.

        \bObvykle nemusíš přesně říkat, jak má postava něco nést či držet, takže
        si nemusíš dělat starosti, která ruka který předmět drží apod. Někdy je
        ale prospěšné položit jeden objekt do jiného nebo na jiný, např. POLOŽ
        KNIHU DO NÁKUPNÍHO KOŠÍKU nebo POLOŽ VÁZU NA STŮL. Pokud se náruč
        postavy zaplní, může pomoci odložení některých nesených předmětů do
        přepravky jako ve skutečném světě, kdy uneseš více věcí, když jsou v
        batohu nebo v krabici.

        \bČasto získáš mnoho dalších informací (a někdy důležitých vodítek)
        prozkoumáním objektů, které uděláš příkazem PODÍVEJ SE NA (nebo
        jednoduše PROZKOUMEJ, které můžeš zkrátit na P, např. P OBRAZ). Většina
        zkušených hráčů má ve zvyku v nové lokaci prozkoumat všechno v jednom
        sledu. ";
    }

    /* show the Conversation chapter */
    showConversationChapter()
    {
        "\b<b>Interakce s ostatními postavami</b>
        \bTvá postava může v příběhu narazit na další lidi nebo stvůry. S
        těmito postavami můžeš občas komunikovat.\b";

        /* show the customizable conversation instructions */
        conversationInstructions;
    }

    /* Time chapter */
    showTimeChapter()
    {
        "\b<b>Čas</b>";

        if (isRealTime)
        {
            "\bČas v příběhu plyne v reakci na příkazy, které zadáváš. Navíc
            ale některé části příběhu se mohou odkrývat v <q>reálném čase,</q>
            což znamená, že události mohou nastat i během doby, co přemýšlíš o
            svém dalším tahu.

            \bPokud potřebuješ zastavit hodiny, zatímco nebudeš chvíli sedět u
            počítače (nebo jen potřebuješ více času k přemýšlení), můžeš napsat
            PAUZA.";
        }
        else
        {
            "\bČas v příběhu plyne jen v reakci na příkazy, které zadáváš. To
            znamená, že se nic nestane během doby, co příběh čeká, až něco
            napíšeš. Každý příkaz v příběhu zabere přibližně stejnou dobu. Pokud
            přímo chceš, aby v příběhu uplynulo trochu času navíc, protože
            očekáváš, že se něco stane, můžeš napsat ČEKEJ (nebo jen Č). ";
        }
    }

    /* Saving, Restoring, and Undo chapter */
    showSaveRestoreChapter()
    {
        "\b<b>Ukládání a načítání</b>
        \bV kterýkoliv okamžik můžeš uložit obraz současné pozice v příběhu,
        abys mohl později se ke stejné pozici vrátit. Pozice bude zapsána do
        souboru na disku počítače. Různých pozic si můžeš uložit, kolik budeš
        chtít (kolik ti dovolí volná kapacita disku).\b";

        switch (crueltyLevel)
        {
        case 0:
            "V tomto příběhu tvá postava nebude nikdy zabita a ani se nemůžeš
            nijak dostat do sitace, ve které by nebylo možné příběh dokončit.
            Cokoliv se tvé postavě stane, nebude bránit nalezení cesty k
            dokončení příběhu. Takže na rozdíl od mnoha jiných textových her si
            nemusíš dělat hlavu s ukládáním pozicí, aby ses ochránil proti
            ocitnutí se v neřešitelné situaci. Samozřejmě i přesto si můžeš
            pozici ukládat tak často, jak budeš chtít, abys mohl odejít a
            jindy se vrátit, nebo si uložit určité pozice, které budeš chtít
            později znovu navštívit.";
            break;

        case 1:
        case 2:
            "V příběhu se může stát, že tvá postava bude zabita nebo že zjistíš,
            že se nacházíš v neřešitelné situaci, ze které není možné dokončit
            příběh. Takže by sis měl <<crueltyLevel == 1 ? 'čas od času' :
            'často'>> pozici ukládat, aby ses nemusel vracet zpátky příliš
            daleko, kdyby to nastalo. ";

            if (crueltyLevel == 2)
                "(Měl bysis zvyknout ponechávat si i všechny staré uložené
                pozice, protože nemusíš hned poznat, že jsi se dostal do
                situace, ze které už nejde hru dokončit. Občas se může stát,
                že se budeš muset vrátit zpátky více než k poslední pozici,
                o které sis <i>myslel</i>, že je bezpečná.)";

            break;
        }

        "\bK uložení pozice napiš ULOŽ na příkazovém řádku. Příběh se zeptá na
        jméno souboru na disku, do kterého se obraz uloží. Musíš zadat takové
        jméno, které je vhodné pro tvůj počítač a na disku musí být dostatek
        volného místa. Pokud by nastal nějaký problém s ukládáním, počítač ti to
        oznámí. Měl bys použít takový název souboru, který ještě na tvém stroji
        neexistuje, protože jinak nový soubor přepíše existující soubor stejného
        jména.

        \bDříve uloženou pozici můžeš obnovit zadáním NAČÍST na příkazovém
        řádku. Příběh se zeptá na název souboru k obnovení. Po načtení souboru
        počítačem bude všechno v příběhu v přesně té situaci, v jaké to bylo
        v době uložení.";

        "\b<b>Vracení tahů</b>
        \bDokonce, i když jsi svou pozici v poslední době neuložil, můžeš obvykle
        odvolat posledních několik tahů příkazem VRAŤ TAH. Pokaždé, když
        zadáš VRAŤ TAH, příběh anuluje efekt jednoho příkazu a vrátí se do stavu
        před zadáním toho příkazu. VRAŤ TAH je limitovaný na vrácení jen
        posledních několika tahů, takže není náhradou za ULOŽ/NAČTI, ale hodí
        se, pokud se tvá postava z nenadání ocitne v nebezpečné situaci nebo
        uděláš chybu, kterou chceš vzít zpátky.";
    }

    /* Other Special Commands chapter */
    showSpecialCmdChapter()
    {
        "\b<b>Další speciální příkazy</b>
        \bPříběh rozumí několika dalším zvláštním příkazům, které by se ti mohly
        hodit.

        \bZNOVU (nebo jen ZN): Zopakuje poslední příkaz. (Pokud v posledním
        zadání bylo několik příkazů najednou, zopakován bude jen ten poslední
        z nich.)
        \bINVENTÁŘ (nebo jen I): Zobrazí, co tvá postava nese s sebou.
        \bROZHLÉDNI SE (nebo jen R): Ukáže úplný popis lokace, ve které se tvá
        postava nachází.";

        /* if the exit lister module is active, mention the EXITS command */
        if (gExitLister != nil)
            "\bSMĚRY: Zobrazí seznam očividných směrů vedoucích ze současné
            lokace.
            \bVYPNOUT/ZAPNOUT SMĚRY, SMĚRY V ZÁHLAVÍ/POPISU: Ovlivňuje způsob,
            jakým je zobrazován seznam směrů vedoucích z lokace. ZAPNOUT SMĚRY
            dá seznam do stavové řádky a také bude vypisovat směry s každým
            popisem lokace. VYPNOUT SMĚRY vypne oba způsoby zobrazení. SMĚRY V
            ZÁHLAVÍ zapne seznam jen ve stavové řadce a SMĚRY V POPISU zapne
            zobrazení jen s popisky lokací.";
        
        "\bOPRAVA (nebo jen O): Napraví jedno špatně napsané slovo v příkazu bez
        nutnosti znovu zadat celý příkaz. Příkaz můžeš použít bezprostředně po
        tom, co tě příběh upozorní na neznámé slovo z předchozího příkazu. Zadej
        příkaz OPRAVA a hned za něj opravené slovo.
        \bKONEC (nebo jen K): Ukončí příběh.
        \bRESTARTOVAT: Spustí příběh znovu od začátku.
        \bNAČTI: Obnoví pozici dříve uloženou příkazem ULOŽ.
        \bULOŽ: Uloží současnou pozici do souboru na disku.
        \bZÁPIS: Začne zaznamenávat zápis průběhu sezení. Veškerý zobrazený text
        bude uložen do souboru, aby sis ho mohl později prostudovat nebo
        vytisknout.
        \bVYPNOUT ZÁPIS: Ukončí zápis průběhu sezení, který jsi zahájil příkazem
        ZÁPIS.
        \bVRAŤ TAH: Odvolá poslední provedený příkaz.
        \bULOŽ NASTAVENÍ: Uloží současné nastavení příkazů UPOZORŇOVÁNÍ, SMĚRY a
        POZNÁMKY jako výchozí. To znamená, že tato nastavení se automaticky
        projeví, až příště spustíš novou hru nebo restartuješ současnou.
        \bNAČTI NASTAVENÍ: Explicitně obnoví nastavení do stavu, který jsi dříve
        uložil příkazem ULOŽ NASTAVENÍ. ";
    }
    
    /* Unknown Words chapter */
    showUnknownWordsChapter()
    {
        "\b<b>Neznámá slova</b>
        \bPříběh se nesnaží předstírat, že zná každé české slovo. Dokonce může
        občas použít slova v popisu, kterým sám nerozumí v příkazech. Když
        napíšeš příkaz obsahující slovo, které příběh nezná, tak ti řekne,
        kterému slovu nerozumí. Pokud příběh nezná slovo pro něco, co popisoval
        a nezná ani žádná synonyma pro tuto věc, můžeš předpokládat, že objekt
        je tu jen jako detail scény, ale pro příběh není důležitý. ";
    }

    /* Ambiguous Commands chapter */
    showAmbiguousCmdChapter()
    {
        "\b<b>Nejednoznačné příkazy</b>
        \bKdyž zadáš příkaz, ve kterém budou chybět některé důležité informace,
        příběh se pokusí ze všech sil přijít na to, co máš na mysli. Bude-li z
        kontextu dostatečně zřejmé, co jsi chtěl udělat, sám si chybějící
        informace doplní a příkaz vykoná. V těchto případech tě upozorní na to,
        co si domyslel, aby předešel nedorozumění v tom, co předpokládá příběh
        a co ty. Například:

        \b
        \n\t &gt;UVAŽ PROVAZ
        \n\t (k háku)
        \n\t Provaz je teď uvázán k háku. Konec provazu
        \n\t se téměř dotýká dna jámy.
        
        \bPokud je však příkaz příliš vágní a příběh nedokáže spolehlivě
        odhadnout tvé přání, zeptá se tě na další informace. Na otázku můžeš
        odpovědět tak, že napíšeš chybějící informaci.

        \b
        \n\t &gt;ODEMKNI DVEŘE
        \n\t Čím je chceš odemknout?
        \b
        \n\t &gt;KLÍČEM
        \n\t A který klíč máš na mysli? Zlatý klíč, nebo stříbrný klíč?
        \b
        \n\t &gt;ZLATÝ
        \n\t Odemčeno.

        \bPokud se příběh ptá na podobnou doplňující otázku a ty se rozhodneš,
        že vlastně nechceš pokračovat s původním příkazem, můžeš místo odpovědi
        napsat zcela nový příkaz.";
    }

    /* Advance Command Formats chapter */
    showAdvancedCmdChapter()
    {
        "\b<b>Složitější formáty příkazů</b>
        \bJakmile si zvykneš na zadávání příkazů, mohly by tě zaujmout některé
        složitější formáty příkazů, kterým příběh rozumí. Všechny tyto pokročilé
        příkazy jsou zcela nepovinné, protože to samé lze vždy provést
        jednoduššími způsoby, kterými jsme se doteď zabývali. Zkušení hráči ale
        mívají pokročilé formáty v oblibě, protože jim šetří psaní.

        \b<b>Použití více objektů najednou</b>
        \bVe většině příkazů můžeš pracovat s více objekty v rámci jednoho
        příkazu. Použij spojku A nebo čárku k oddělení jednoho objektu od
        druhého:

        \b
        \n\t VEM KRABICI, DISKETU A PROVAZ
        \n\t DEJ DISKETU A PROVAZ DO KRABICE
        \n\t POLOŽ KRABICI A PROVAZ
        
        \bMůžeš použít slova VŠE nebo VŠECHNO k odkázání se na každý použitelný
        objekt ve svém příkazu a také můžeš použít slova KROMĚ nebo MIMO
        (hned za slovem VŠE) k vyloučení určitých objektů:

        \b
        \n\t VEZMI VŠE
        \n\t DEJ VŠE KROMĚ DISKETY A PROVAZU DO KRABICE
        \n\t VYNDEJ VŠECHNO Z KRABICE
        \n\t VEM VŠE Z POLICE

        \bVŠE odkazuje na všechno, co má v rámci příkazu smysl, mimo věcí uvnitř
        jiných objektů, které jsou zahrnuty slovem VŠE. Například pokud neseš
        kracici a provaz a krabice obsahuje disketu, napsáním POLOŽ VŠE dojde k
        odložení krabice a provazu, ale disketa zůstane uvnitř krabice.

        \b<b><q>Ho</q>, <q>ji</q> a <q>je</q></b>
        \bNa poslední použitý objekt nebo objekty se můžeš odkázat zájmeny HO,
        JI a JE:

        \b
        \n\t VEM KRABICI
        \n\t OTEVŘI JI
        \n\t VEM DISKETU A PROVAZ
        \n\t DEJ JE DO KRABICE

        \b<b>Více příkazů najednou</b>
        \bDo jedné příkazové řádky můžeš vložit více příkazů. Oddělíš je od sebe
        tečkou, slovem POTOM, čárkou nebo spojkou A. Například:

        \b
        \n\t VEM DISKETU A DEJ JI DO KRABICE
        \n\t VEZMI KRABICI. OTEVŘI JI.
        \n\t ODEMKNI DVEŘE KLÍČEM. OTEVŘI JE, POTOM JDI NA SEVER.

        \b Pokud příběh nerozumí některému z příkazů, tak dá vědět, čemu
        nerozumí a bude ignorovat všechno ostatní na řádce směrem doprava.";
    }

    /* General Tips chapter */
    showTipsChapter()
    {
        "\b<b>Několik tipů</b>
        \bTeď, když už znáš technické pozadí zadávání příkazů, mohlo by tě
        zajímat pár tipů k strategii. Níže najdeš několik postupů, které
        zkušní hráči interaktivní fikce používají při hře.

        \bVšechno PROZKOUMEJ, zlášť když poprvé vstoupíš do nové lokace. Bližší
        pohled na jednotlivé objekty většinou odhalí detaily, které nebyly
        zmíněny v širším popisu lokace. Pokud prozkoumání objektu zmíní jeho
        detailní části, prozkoumej je také.

        \bPokud má příběh více než jen pár lokací, tak si kresli mapu. Když se
        poprvé dostaneš do nové lokace, poznamenej si všechny směry. Budeš mít
        pak lepší přehled, zda tu nejsou ještě nějaké oblasti, které jsi doposud
        neprošel. Když se zasekneš, můžeš v mapě najít neprobádaná území a v
        nich se může skrývat, co jsi hledal.

        \bPokud příběh některému slovu a ani jeho synonymům nerozumí, je
        pravděpodobné, že objekt, se kterým se snažíš manipulovat, není v příběhu
        důležitý. Pokud se snažíš něčím manipulovat a příběh řekne něco jako
        <q>to není duležité,</q> pak asi můžeš objekt jednoduše ignorovat,
        protože je tu nejspíše jako součást scenérie, aby scéna byla
        detailnější.

        \bSnažíš-li se něčeho dosáhnout a nic nevypadá, že by zabíralo, věnuj
        pozornost tomu, co se děje špatně. Pokud vše, co zkoušíš, se setkává s
        naprostým odmítnutím (<q>nic se nestalo</q> nebo <q>to není něco, co bys
        mohl otevřít</q>), budeš asi na špatné stopě. Udělej si odstup a zamysli
        se, jak k problému přistoupit z jiného úhlu pohledu. Pokud odpovědí je
        něco více specifického, může to být vodítko. <q>Strážce říká: <q>To tu
        nemůžeš otevřít!</q>\ a vytrhl ti krabičku z rukou</q> &mdash;
        to může například naznačovat, že máš donutit strážce odejít nebo že máš
        krabičku vzít někam jinam, než ji otevřeš.

        \bKdyž se úplně zasekneš, zkus současný hlavolam chvíli nechat stranou a
        pracovat na jiném problému. Dokonce může být prospěšné
        pozici uložit a dát si od hry přestávku. Řešení problému, který maří tvůj
        postup, ti může znenadání problesknout hlavou, když ho nejméně čekáš, a
        takové okamžiky poznání jsou nejvíce uspokojující. Některé příběhy jsou
        nejzáživnější, když je hraješ několik dní, týdnů nebo dokonce i
        měsíců; když si příběh užíváš, proč jím prochvátat?

        \bA nakonec, pokud všechno selže, můžeš zkusit požádat o nápovědu.
        Skvělé místo, kam si zajít pro radu, je server
        <a href='http://www.textovky.cz/'>www.textovky.cz</a> a jeho fórum. ";

        "\n";
    }

    /* INSTRUCTIONS doesn't affect UNDO or AGAIN */
    isRepeatable = nil
    includeInUndo = nil
;


/* ------------------------------------------------------------------------ */
/*
 *   define the INSTRUCTIONS command's grammar 
 */
VerbRule(instructions) 'instrukce' | 'manuál' | 'návod' | 'pomoc' | 'slovník':
    InstructionsAction
;


/* ------------------------------------------------------------------------ */
/*
 *   The instructions, rendered in menu form.  The menu format helps break
 *   up the instructions by dividing the instructions into chapters, and
 *   displaying a menu that lists the chapters.  This way, players can
 *   easily go directly to the chapters they're most interested in, but
 *   can also still read through the whole thing fairly easily.
 *   
 *   To avoid creating an unnecessary dependency on the menu subsystem for
 *   games that don't want the menu-style instructions, we'll only define
 *   the menu objects if the preprocessor symbol INSTRUCTIONS_MENU is
 *   defined.  So, if you want to use the menu-style instructions, just
 *   add -D INSTRUCTIONS_MENU to your project makefile.  
 */
#ifdef INSTRUCTIONS_MENU

/* a base class for the instruction chapter menus */
class InstructionsMenu: MenuLongTopicItem
    /* 
     *   present the instructions in "chapter" format, so that we can
     *   navigate from one chapter directly to the next 
     */
    isChapterMenu = true

    /* the InstructionsAction property that we invoke to show our chapter */
    chapterProp = nil

    /* don't use a heading, as we provide our own internal chapter titles */
    heading = ''

    /* show our chapter text */
    menuContents = (InstructionsAction.(self.chapterProp)())
;

InstructionsMenu template 'title' ->chapterProp;

/*
 *   The main instructions menu.  This can be used as a top-level menu;
 *   just call the 'display()' method on this object to display the menu.
 *   This can also be used as a sub-menu of any other menu, simply by
 *   inserting this menu into a parent menu's 'contents' list.  
 */
topInstructionsMenu: MenuItem 'Jak hrát interaktivní fikci';

/* the chapter menus */
+ MenuLongTopicItem
    isChapterMenu = true
    title = 'Úvod'
    heading = nil
    menuContents()
    {
        "\b<b>Úvod</b>
        \bVítej! Tento návod ti pomůže začít, pokud jsi ještě žádný příběh
        interaktivní fikce nehrál. Jestli ale už víš, jak se podobné hry hrají,
        můžeš samozřejmě čtení všech instrukcí přeskočit a raději zadat na
        příkazové řádce příkaz O HŘE, kde se v souhrnu dozvíš detaily o tomto
        konkrétním příběhu.
        \b
        Aby se v návodu lépe hledalo, je rozdělen do několika kapitol. ";

        if (curKeyList != nil && curKeyList.length() > 0)
            "Na konci každé kapitoly zmáčkni
            <b><<curKeyList[M_SEL][1].toUpper()>></b> k posunu na
            další kapitolu nebo <b><<curKeyList[M_PREV][1].toUpper()>></b>
            k návratu do obsahu. ";
        else
            "K listování kapitolami klikej na odkazy nebo použij klávesy šipek
            doleva a doprava. ";
    }
;

+ InstructionsMenu 'Zadávání příkazů' ->(&showCommandsChapter);
+ InstructionsMenu 'Zkracování příkazů' ->(&showAbbrevChapter);
+ InstructionsMenu 'Pohyb' ->(&showTravelChapter);
+ InstructionsMenu 'Manipulace s předměty' ->(&showObjectsChapter);
+ InstructionsMenu 'Interakce s ostatními postavami'
    ->(&showConversationChapter);
+ InstructionsMenu 'Čas' ->(&showTimeChapter);
+ InstructionsMenu 'Ukládání a načítání' ->(&showSaveRestoreChapter);
+ InstructionsMenu 'Další speciální příkazy' ->(&showSpecialCmdChapter);
+ InstructionsMenu 'Neznámá slova' ->(&showUnknownWordsChapter);
+ InstructionsMenu 'Nejednoznačné příkazy' ->(&showAmbiguousCmdChapter);
+ InstructionsMenu 'Složitější formáty příkazů' ->(&showAdvancedCmdChapter);
+ InstructionsMenu 'Několik tipů' ->(&showTipsChapter);

#endif /* INSTRUCTIONS_MENU */

