#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena. Podrobnosti o šíření v souboru LICENCE.txt.
 */

#include <adv3.h>
#include <cs_cz.h>

/*
 *   Zavedeme slovník pro abstraktní témata, kterým neodpovídá žádný objekt ve
 *   hře, ale hráč se na ně může chtít vyptávat. Dáváme 1., 4. a 6. pád.
 */
tCargo: Topic 'náklad/nákladu';
tFuelConsumption: Topic 'spotřeba spotřebu spotřebě paliva/palivo/palivu';
tJourney: Topic 'cesta/cesty/cestě/cestu/cestou';
tService: Topic 'převzetí přebrání služby/službu/služba';

/*
 *   Definice postavy kapitána. Základem je třída Person a doplňujeme TourGuide
 *   kvůli společnému výstupu z lodi do základy, kdy nás kapitán vede, a
 *   zvláštní třídu ConversationStateSync, která zajišťuje, aby v jednu chvíli
 *   v kajutě velitele probíhal rozhovor s oběma postavami najednou, jako by
 *   byli oba jednou postavou.
 */
captain: ConversationStateSync, TourGuide, Person
    'henry scott/kapitán' 'kapitán' *1 @cPilotChair

    descCount = 0
    desc()
    {
        "Kapitán je starší mírně plešatící chlapík s prošedivělými vlasy a
            několikadenním strništěm. ";

        /*
         *   Delší popisek by byl časem byl otravný, tak ho po dvou vypsáních
         *   či po přistání zkrátíme.
         */
        if(++descCount <= 2 && !gRevealed('ship-landed'))
            "Kdysi sloužil jako vojenský pilot, ale nakonec už nebyl schopen
                plnit přísné kvalifikační požadavky a jako mnoho jiných v jeho
                věku byl nucen volit mezi pilotáží méně prestižních strojů nebo
                úplným odchodem ze služby. Na Zemi ani na některou z kolonií ho
                nic netáhlo, proto teď velí téhle konzervě. ";
    }

    /* Na začátku hry sedí kapitán v křesle pilota, viz výše @cPilotChair. */
    posture = sitting
    standUp()
    {
        inherited();
        "Kapitán vstává a odchází z pilotní kabiny. ";
    }

    /*
     *   Konfigurace třídy ConversationStateSync, která zajišťuje synchronizaci
     *   konverzačních stavů obou postav v kajutě.
     */
    otherActor = commander
    myConversationReadyState = captainPaperWork
    myInConversationState = captainPaperWorkTalking
    otherConversationReadyState = commanderPaperWork
    otherInConversationState = commanderPaperWorkTalking

    /*
     *   Na konci hry je kapitán i velitel ve vozítku. Pokud hráč chce střílet
     *   na postavu, tak vlastně tím myslí vozítko.
     */
    dobjFor(ShootAt)
        maybeRemapTo(me.isIn(tunnel), ShootAt, secondRover, IndirectObject)
    dobjFor(Attack)
        maybeRemapTo(me.isIn(tunnel), Attack, secondRover)

    /*
     *   Ve slovníku jsou přivlastňovací přídavná jména postav (např. kapitánův)
     *   označena podle anglické zvyklosti apostrofem s písmenkem s. Knihovna je
     *   podle toho rozpozná a pamatuje si, že jsou přivlastňovací.
     */
    gcName = 'kapitána, kapitánovi, kapitána, kapitánovi, kapitánem'
    gcVocab = 'henryho henrymu henrym kapitánův\'s kapitánova\'s kapitánovo\'s
        scotta/scottovi/scotte/scotta/scottovi/scottem/kapitána/kapitáne/
        kapitánovi/kapitánem'
;

/*
 *   Kapitánův tablet je zmíněn v atmosféře na začátku hry, tak aby šel
 *   prozkoumat. Decoration na vše kromě zkoumání reaguje slovy, že není
 *   důležité.
 */
+ Decoration
    'tablet/displej/display' 'kapitánův tablet' *2
    "Je úplně stejný, jako tvůj, jen barvu má mnohem hezčí. Asi aby si ho
        kapitán snadno poznal. "

    /*
     *   Když parser ví, že je kapitánův, může ho rozlišit v otázce "a který máš
     *   na mysli" právě podle vlastnictví. Viz přivlastňovací přídavná jména ve
     *   slovníku postavy.
     */
    owner = captain

    /*
     *   Když jsme z namíření tabletu na robota udělali klíčový puzzle, tak holt
     *   musíme být konsistentní a reagovat při míření na kdejakou elektronickou
     *   pitomost.
     */
    iobjFor(AimAt)
    {
        action()
        {
            "Namířil jsi svůj tablet na kapitánův. Kamerka tabletu rozpoznala
                jiný tablet a tvůj tablet se pokusil vyvolat chatovací dialog,
                ale kapitán ho má zakázaný, abys ho nezdržoval jako posledně. ";
        }
    }

    gcName = 'kapitánova tabletu, kapitánově tabletu, kapitánův tablet,
        kapitánově tabletu, kapitánovým tabletem'
    gcVocab = 'displeje displeji displejem displaye displayi displayem tabletu/
        tabletem'
;

/* ------------------------------------------------------------------------ */
/*
 *   Nejprve globání témata konverzace přidaná přímo do postavy. Ty jsou k
 *   dispozici vždy, pokud není řečeno jinak pomocí isActive.
 */
+ AskTellTopic @commander
    "<.p><q>Kapitáne?</q> zeptal ses, <q>je velitel váš dobrý známý?</q>

        <.p><q>Jo,</q> houkl na tebe kapitán a jeho stažené obočí ti odseklo
        další dotazy. "
;

+ AskTellTopic @technician
    "<.p><q>Kapitáne,</q> obrátil se k svému šéfovi, <q>jak znáte toho jejich
        technika? Mohu mu věřit?</q>

        <.p><q>Co by, technik,</q> pokrčil kapitán rameny, <q>nikdy jsem ho
        nepotřeboval, prohodíme občas pár vět u jídla.</q> "
;

/*
 *   Pokud postava nemá odpověď, řekně nějakou z výmluv níže. Rozlišíme výmluvy
 *   při otázce a při ostatních konverzačních příkazech, kdy jsou trochu jinak
 *   formulované. ShuffledEventList po vyčerpání odpovědí je znovu zamíchá.
 *   Reveal default-topic zařídí zobrazení tipu tutoriálu.
 */
+ DefaultAskTopic, ShuffledEventList
    [
        '<.p>Pokusil ses oslovit kapitána, ale ten zívá a vůbec tě nevnímá.
            <.reveal default-topic> ',
        '<.p><q>Kapitáne, mohl bych…?</q> začal jsi mluvit, ale kapitán tě
            otráveně přerušil:
            <.p><q>Co jsi říkal?</q><.reveal default-topic> ',
        '<.p>Nadechl ses: <q>Pane, rád bych…,</q> ale kapitán ti nedal domluvit:
            <.p><q>Dej pokoj, nemusíš vědět všechno.</q>
            <.reveal default-topic> ',
        '<.p><q>Můžu se zeptat?</q> obrátil ses ke kapitánovi.
            <.p><q>Kdo se moc ptá, moc se dozví,</q> reaguje na tvou otázku
            kapitán.<.reveal default-topic> ',
        '<.p><q>Pane,</q> ale než jsi zformuloval další slova, kapitán se
            ušklíbl:
            <.p><q>Nebuď zvědavej, budeš brzo starej.</q>
            <.reveal default-topic> '
    ]
;

+ DefaultAnyTopic, ShuffledEventList
    [
        '<.p>Pokusil ses oslovit kapitána, ale ten zívá a vůbec tě nevnímá.
            <.reveal default-topic> ',
        '<.p><q>Kapitáne, mohl bych…?</q> začal jsi mluvit, ale kapitán tě
            otráveně přerušil:
            <.p><q>Co jsi říkal?</q><.reveal default-topic> '
    ]
;

/* ------------------------------------------------------------------------ */
/*
 *   Celá tato skupina je k dispozici v jídelně. Řešíme jako TopicGroup, protože
 *   v jídelně to jsou dva samostatné stavy, do kterých mají témata patřit.
 *   TopicGroup aplikuje společné isActive na vše, co v něm je obsaženo.
 */
+ TopicGroup
    isActive = me.isIn(canteen)
;

/* Tady reagujeme na kterýkoliv ze seznamu objektů, proto hranaté závorky. */
++ AskTellTopic [tDrink, platter, strawberries, pasta]
    "<.p><q>Chutná?</q> zeptal ses jen tak do pléna. Kapitán nejdříve polknul a
        teprve pak dal na vědomí, že tvůj dotaz slyšel:
        <.p><q>Hmm, ještě jiné blbé dotazy?</q> "
;

++ AskTellTopic @sq71
    "<.p>Polknul jsi sousto a obrátil ses k pánům: <q>Tak jak proběhla inspekce?
        Kde jste vlastně byli?</q>

        <.p>Kapitán pokrčil rameny, pohlédl na velitele a zamručel: <q>Hmm,
        normálka, klasická obhlídka nejbližšího okolí.</q> Vložil jsi do úst
        další sousto a přemýšlel, proč popírají dlouhý výlet, který jsi viděl na
        navigaci. Co tam asi je? Ale od nich se to asi nedozvíš. "
;

++ AskTellAboutForTopic [card, rover]
    "<.p><q>Dal by se udělat výlet vozítkem?</q> zkusil jsi nenápadně nadhodit.
        <.p>Kapitán nejdříve polknul a teprve pak dal na vědomí, že tvůj dotaz
        slyšel: <q>Máš málo práce, že přemýšlíš o hloupostech?</q> "
;

++ AskTellTopic [tCmx12, tRab4]
    "<.p><q>Takže co tu vlastně používáte za rypadla?</q> obrátil ses s dotazem
        na velitele, <q>CMX nebo RA?</q>
        <.p>Za chvilku jsi svého dotazu litoval. Velitel tě zavalil spoustou nic
        neříkajících čísel, takže ses stejně nic nedověděl. "
;

/* ------------------------------------------------------------------------ */
/*
 *   ROZHOVOR V KABINĚ PŘED NEHODOU
 *
 *   Dvojice InConversationState a ConversationReadyState spolu s HelloTopic a
 *   ByeTopic zajišťuje navazování a odchod z konverzace. Vloží se do postavy v
 *   tomto uspořádání.
 */
+ captainTalking: InConversationState

    /*
     *   U stavu postavy se zadávají dva popisky. Zatímco specialDesc se zobrazí
     *   v samostatném odstavci po rozhlédnutí se v místnosti, stateDesc se
     *   připojí na konec běžného prozkoumání postavy, a proto obvykle
     *   neobsahuje už podmět, protože jméno postavy už je zřejmé z předchozích
     *   vět.
     */
    stateDesc = "Baví se s tebou. "
    specialDesc = "Kapitán sedí v pilotním křesle a baví se s tebou. "

    /* Pokud už je čas, tak po navázání rozhovoru nabídneme převzetí služby. */
    activateState(actor, oldState)
    {
        /*
         *   Vždy když předefinujeme metodu knihovny, musíme zavolat zděděné
         *   chování, leda bychom ho chtěli úmysleně potlačit.
         */
        inherited(actor, oldState);

        /*
         *   Jakmile zahájíme rozhovor po té, co jsou splněny podmínky pro
         *   převzetí služby, nabídneme konverzační uzel, ve kterém může hráč
         *   službu převzít a zobrazíme také tip tutoriálu.
         */
        if(gRevealed('treadmill-run') && tube.eated)
        {
            captain.setConvNode('me-driving');
            gReveal('special-topic');
        }
    }
;

++ captainDriving: ConversationReadyState

    /* V tomto stavu má být postava na začátku hry. */
    isInitState = true

    stateDesc = "Loď je ve skutečnosti po většinu času řízena autopilotem, takže
        toho k práci moc není, jen dozor nad přístroji. "

    specialDesc = "Kapitán sedí v pilotním křesle a řídí loď. "
;

/* StopEventList prochází možnosti v pořadí a poslední pak opakuje dokola. */
+++ HelloTopic, StopEventList
    [
        '<.p><q>Brýtro, kapitáne,</q> pozdravil jsi <<if
            !treadmill.achievement.scoreCount &&
            !kitchen.achievement.scoreCount>>rozespale<<else if
            treadmill.achievement.scoreCount &&
            !kitchen.achievement.scoreCount>>udýchaně<<else>>spokojeně<<end>>.
            <.p>Kapitán se k tobě otočil. <q>Á, konečně jsi vzhůru. Už jsem tě
            chtěl jít z toho pytle vysypat.</q> ',
        '<.p><q>Kapitáne?</q> naklonil ses ke kapitánovi.
            <.p>Kapitán unaveně zvedl hlavu: <q>Co potřebuješ?</q> '
    ]
;

/*
 *   ShuffledEventList projde všechny možnosti v náhodném pořadí a po vyčerpání
 *   se znovu zamíchá.
 */
+++ ByeTopic, ShuffledEventList
    [
        '<.p><q>Tak já jdu ještě něco udělat.</q>
            <.p>Kapitán obrátil svou pozornost zpět k palubním přístrojům. ',
        '<.p><q>Hned jsem zpátky.</q>
            <.p>Kapitán obrátil svou pozornost zpět k palubním přístrojům. ',
        '<.p><q>Sorry, musím odběhnout.</q>
            <.p>Kapitán obrátil svou pozornost zpět k palubním přístrojům. ',
        '<.p><q>Moment. Vrátím se.</q>
            <.p>Kapitán obrátil svou pozornost zpět k palubním přístrojům. '
    ]
;

+++ ImpByeTopic
    "Kapitán se opět plně věnuje své práci. "
;

/* Pár běžných témat, která jsou vložená do přislušného InConversationState. */
++ AskTellTopic, SuggestedAskTopic @ship
    "<.p><q>Proč to tu udělali tak mrňavé?</q> postěžoval sis při pokusu
        protáhnout si záda.

        <.p><q>Jako by někdy společnosti záleželo na lidech, pro ně jsou hlavní
        prachy, naše pohodlí je až na posledním místě,</q> popíchnul tě kapitán
        stokrát opakovaným rozhovorem. "

    /*
     *   Jestliže nastavíme téma jako doporučené (SuggestedAskTopic), aby se na
     *   něj hráč zeptal, musíme napsat, jak bude téma označené v nabídce, tj.
     *   uvedeme název objektu ve čtvrtém pádě (zeptat se na koho/co). V případě
     *   SuggestedTellTopic pak mohl bys mu říci o kom/čem 6. pád.
     */
    name = 'loď'
;

++ AskTellTopic, SuggestedAskTopic, StopEventList @tJourney
    [
        '<.p><q>Tak jak to odsejpá? Škubalo to mnou i v pelechu,</q> přidržel
            ses opěradla.

            <.p><q>Za chvíli tam budem, už je tu spousta blbejch šutrů, co se
            motaj do cesty,</q> unaveně povzdechl kapitán. '
    ]

    name = 'cestu'
;

++ AskTellTopic, SuggestedAskTopic, StopEventList [asteroid, base]
    [
        '<.p><q>Už je ten šutrák vidět?</q> <<if me.isIn(cockpit)>>naklonil
            ses zvědavě k průzoru<<else>>zeptal ses z povzdálí<<end>>,
            <q>ty lidi obdivuju -- je jich pět a půl tak daleko od Země jen
            kvůli tomu, že je tam zajímavá koncentrace surovin. A jediné
            zpestření je náš přílet. By mě zajímalo, co by dělali, kdybychom
            nedorazili.</q>

            <.p><q>Zas tak daleko to není,</q> zamručel od řízení kapitán a ani
            se po tobě neohlédl, <q>u armády jsme lítali mnohem dál než do
            asteroidového pásu mezi Marsem a Jupiterem.</q> '
    ]

    name = 'asteroid'
;

++ AskTellTopic, SuggestedAskTopic, StopEventList [tCargo, tFuelConsumption]
    [
        '<.p><q>Myslel jsem, že letíme skoro s prázdnou,</q> řekl jsi s pohledem
            upřeným na obrazovky, <q>a podle tohoto ukazatele snad stěhujeme
            slona.</q>

            <.p>Kapitán poklepal na obrazovku a odfrkl si: <q>Zatracená
            kalibrace!</q> ',

        '<.p>Znovu jsi upřel pohled na ukazatele spotřeby: <q>Asi bych se na to
            měl na základně podívat. Ale v každém případě bychom to měli dát do
            záznamu, ať se na to podívají doma technici.</q>

            <.p><q>To neřeš,</q> zabručel kapitán, <q>tahle mašina to dělá při
            každé cestě, po každé opravě. Furt.</q> ',

        '<.p>Kapitánovi jsi o zvýšené spotřebě paliva už řekl, ale prý si s tím
            nemáš dělat žádné starosti. '
    ]

    name = 'náklad'

    /*
     *   Protože tohle doporučené téma je na pokračování, tj. jde o něm mluvit
     *   déle, potom je vhodné říci, kolikrát má být doporučeno. V tomto případě
     *   o jedna méně, než je počet odpovědí, protože poslední odpověď už jen
     *   říká, že postava nic dalšího neví. V tom momentě už téma nebudeme
     *   doporučovat. Navíc poslední odpověď označíme, že není konverzační, aby
     *   se v případě, že by k ní došlo, neodehrálo pozdravení.
     */
    timesToSuggest = 2
    isConversational = !curiositySatisfied
;

/*
 *   Hráč si musí nejprve zaběhat na pásu a ohřát sendvič ke snídani, aby se
 *   naučil hru ovládat. Do té doby mu vysvětlujeme, že na převzetí služby je
 *   ještě čas.
 */
++ AskTellTopic, SuggestedAskTopic @tService
    "<.p><q>Jak jsme na tom s časem?</q> zkontroloval jsi časový údaj, <q>mám
        vás už vystřídat?</q>

        <.p><q>Ještě tu chvíli vydržím,</q> opáčil kapitán, <q>běž se trochu
        rozhejbat, zacvič si, pak si tohodle křesla užiješ, až tě to bude
        mrzet.</q> "

    name = 'převzetí služby'
;

+++ AltTopic, SuggestedAskTopic
    "<.p><q>Jak jsme na tom s časem?</q> zkontroloval jsi časový údaj, <q>mám
        vás už vystřídat?</q>

        <.p><q>V poho,</q> odvětil kapitán, <q>jsi už po snídani? V klidu se
        najez, ta chvíle už mě nezabije. V ohřívači jsem ti nechal sendvič,
        ale asi bude ještě zmrzlej, tak si ho ohřej.</q> "

    name = 'převzetí služby'

    /*
     *   AltTopic je odpověď, která nahradí výše uvedenou odpověď na @tService,
     *   jakmile je splněna podmínka isActive. V tomto případě po té, co si hráč
     *   zaběhá na pásu.
     */
    isActive = gRevealed('treadmill-run');
;

/* AltTopiců může být více, poslední splňující asActive přebíjí předchozí. */
+++ AltTopic
    "<.p><q>Jak jsme na tom s časem?</q> zkontroloval jsi časový údaj, <q>mám
        vás už vystřídat?</q>

        <.p><q>Ale jo,</q> odvětil kapitán, <q>už bys mohl. Začínám být už
        ospalý. Rád bych byl před přistáním fit.</q> "

    name = 'převzetí služby'
    isActive = gRevealed('treadmill-run') && tube.eated;
;

/*
 *   Až nastane správný okamžik, umožníme hráči převzetí služby v rozhovoru.
 *   Přidáme proto SpecialTopic a ten se smí nacházet jen v konverzačním uzlu,
 *   takže nejprve ConvNode. Uzel necháme "přilepený", než hráč službu převezme,
 *   jinak by po prvním příkazu se z něj zase odešlo.
 */
++ ConvNode 'me-driving' { isSticky = true }

+++ SpecialTopic

    /* Jak označit SpecialTopic v nabídce. */
    name = 'převzít službu'

    /*
     *   SpecialTopic je jednorázová možnost jak v rozhovoru pokračovat, která
     *   se zcela vymyká z běžného schématu zeptej se na/řekni o. Proto musíme
     *   posktnout vyčerpávající slovník, jak hráč může zareagovat. Jakákoliv
     *   odpověď hráče, která se skládá pouze ze slov vyjmenovaných níže, se
     *   bere jako schodující se.
     */
    keywordList = ['převezmi', 'převzít', 'přeber', 'přebrat', 'službu']

    /*
     *   V každém konverzačním objektu můžeme místo odpovědi ve dvojitých
     *   uvozovkách dát metodu topicResponse, ve které toho můžeme udělat více,
     *   než jen napsat text.
     */
    topicResponse()
    {
        /* Hráč při řízení sedí v křesle. */
        me.moveIntoForTravel(cockpit);
        cNavigatorChair.checkMovingActorInto(true);

        /* Nezapomeneme zrušit přilepený konverzační uzel. */
        "<.p><q>Jaká byla šichta? Něco podstatného?</q> připnul sis popruh svého
            křesla a aktivoval obrazovky.

            <.p><q>Uaaauahhh</q> div si zívnutím kapitán nevyvrátil čelisti,
            <q>asteroid už je vidět i bez přístrojů, tak akorát, abych se
            relaxnul, než tam budem. Zobnu nějakou piluli, abych rychle
            usnul.</q><.convnode nil> ";

        /* Kapitána odešleme do své kajuty spát a načasujeme nehodu. */
        captain.setCurState(captainLeaving);
        ship.malfunctionFuse = new Fuse(ship, &sayMalfunction, 3);

        gReveal('me-driving');

        clock.eventReached();
        achievement.awardPointsOnce();
    }
    clock: ClockEvent { eventTime = [1, 21, 2] }
    achievement: Achievement { +1 "převzetí služby" }
;

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN JDE SPÁT
 *
 *   Během cesty se kapitán nenechá rušit, to zařídí HermitActorState. Nechceme
 *   automatické hlášky o pohybu postavy, raději zvolíme vlastní. Cestu řídí
 *   TravelAgendaItem a aktivujese ihned, jak zapneme stav captainLeaving.
 */
+ captainLeaving: HermitActorState
    sayDepartingThroughPassage(conn) { "Zaslechl jsi zabzučení vodomatu a o
        chvilku později klapnutí krytu kapitánovy kóje. "; }
    sayDepartingLocally(traveler, dest) { ""; }
;

+ captainWalkAgenda: TravelAgendaItem
    isReady = (captain.curState == captainLeaving)
    initiallyActive = true
    agendaOrder = 10
    travelPath = [cockpit, livingRoom, captainsCabin]
;

/* ------------------------------------------------------------------------ */
/*
 *   SPOLEČNÝ VÝSTUP Z LODI DO ZÁKLADNY
 *
 *   Kapitán nás vede z vodi do základny, ale na nás je, abychom přesun mezi
 *   lokacemi zahájili. Kapitán jen ukazuje cestu. Každý přechod mezi místnostmi
 *   se skládá z dvojice stavů, jeden pro čekání na nás a druhý pro samotný
 *   přechod mezi lokacemi.
 */
+ captainGuideToDome: GuidedTourState
    stateDesc = "Čeká, až spolu projdete tunelem do základny. "
    specialDesc
    {
        if(!gRevealed('captain-follow'))
            "Kapitán otevřel tlakové dveře lodi: <q>Tak pojď, mladej, už na nás
                čekají.</q><.reveal captain-follow> ";
        else
            "Kapitán stojí u otevřených dveří vedoucích z lodi a čeká, až ho
                budeš následovat. ";
    }

    escortDest = shipDoor
    stateAfterEscort = captainTalkingInDome

    escortStateClass = captainGuideToDomeInTravel
;

class captainGuideToDomeInTravel: GuidedInTravelState
    sayDeparting(conn)
    {
        shipDoor.disable();
        "Následoval jsi kapitána do tunelu. Lehce korigujete pozvolný pád
            rukama, až jste propluli do základny. ";
    }
    specialDesc = "Po žebříku se vynořuje postava, konečně jiná tvář než ta
        kapitánova či ta ze zrcadla, kterou vidíš po dlouhé době letu.
        Odhaduješ, že to je velitel základny, není nejmladší, pohybuje se
        rozvážně, ale očividně má radost z vašeho příletu.
        <<commander.initiateConversation(commanderWelcomes, 'welcome')>>
        <<shipDoor.enable()>> "
;

/* ------------------------------------------------------------------------ */
/*
 *   SETKÁNÍ NA ZÁKLADNĚ
 *
 *   V tomto uzlu není z ConvNodu úniku, takže nepotřebujeme Hello a Bye.
 */
+ captainTalkingInDome: ActorState
    stateDesc = "Hovoří tu s ostatními. "
    specialDesc = "Je tu s tebou kapitán<<if technician.curState ==
        technicianWelcomes>>, velitel a technik<<else>> a velitel<<end>>, kteří
        s tebou hovoří. "
;

/* ------------------------------------------------------------------------ */
/*
 *   PO SKONČENÍ UVÍTÁNÍ
 *
 *   Zatímco od hráče se čeká, že bude následovat Borise do dílny, kapitán
 *   s velitelem si povídají spolu a případně jsou schopni reagovat na běžná
 *   globální témata.
 */
+ captainChatting: ActorState
    stateDesc = "Polohlasem se baví s velitelem o novinkách a soukromých
        záležitostech. "
    specialDesc = "Kapitán s velitelem polohlasem probírají novinky a soukromé
        záležitosti. "
;

++ HelloTopic "Sice už hovoří o něčem jiném, ale klidně na ně můžeš mluvit. ";
++ ByeTopic "<.p><q>Pánové, odcházím zdechnout do houští,</q> obrátil ses k
    veliteli a kapitánovi, <q>teda jdu se podívat na tu vaši dílnu.</q> ";

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN JDE S VELITELEM DO JEHO KAJUTY
 */
+ captainGoToCommandersRoom: HermitActorState
    specialDesc = ""

    /* U HermitActorState se definuje noResponse. */
    noResponse = "Jsou příliš zabraní do společného rozhovoru. "

    /*
     *   Hráč může cestu kapitána a velitele vidět z více různých míst a pokaždé
     *   se jemně liší popis, aby odpovídal úhlu pohledu. Tady je to totiž dost
     *   složité, protože obě části chodby, velín a sklad jsou propojeny
     *   několika různými konektory smyslů, aby bylo vidět z jedné lokace do
     *   další. Proto je tolik možností.
     */
    sayArrivingDownStairs(conn)
    {
        /* Když jsi v západní části chodby a postavy přijdou zeshora. */
        if(conn == ladderUp) "Velitel následován kapitánem sklouzl po žebříku z
            kupole dolů do chodby. ";
    }
    sayDepartingLocally(dest, conn)
    {
        /* Když jsi v západní části chodby a postavy jdou chodbou. */
        if(conn == corridorEast) "Kapitán s velitelem propluli chodbou kolem
            tebe na její vzdálený konec. ";
    }
    sayArrivingLocally(dest, conn)
    {
        /* Když jsi v západní části chodby a postavy jdou k tobě. */
        "Kapitán s velitelem připluli chodbou k tobě. ";
    }
    sayDepartingDir(dir, conn)
    {
        /* Když koukáš z dílny, vydíš jen výsek chodby. */
        if(conn == corridorEast) "Velitel vede kapitána dál chodbou. ";
    }
    sayDepartingThroughPassage(conn)
    {
        /* Když jsi na západním konci chodby. */
        "Kapitán následoval velitele do jeho kajuty. ";
    }
    sayTravelingRemotely(dest, conn)
    {
        "Kapitán s velitelem proplouvají chodbou na její vzdálený konec. ";
    }
;

+ captainGoToCommandersRoomAgenda: TravelAgendaItem
    isReady = (captain.curState == captainGoToCommandersRoom)
    initiallyActive = true
    travelPath = [dome, corridorWest, corridorEast, commandersRoom]
;

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN S VELITELEM VYŘIZUJÍ PAPÍRY V KAJUTĚ
 *
 *   Když kapitán s velitelem dorazí do kajuty, tak se chovají v podstatě jako
 *   jedna postava, čili hráč mluví s oběma najednou a jejich stavy konverzace
 *   se synchronizují. Společná konverzační témata jsou v samostatném souboru
 *   actorCommonTopics.t, Hello a Goodbye mají každý své.
 */
+ captainPaperWorkTalking: ProxyTopicDatabase, InConversationState
    stateDesc = "Odložil papíry, aby se s tebou mohl bavit. "
    specialDesc = "Kapitán s velitelem odložili papíry a jsou zvědaví, s čím jsi
        přišel. "

    /* Dokud hráč neodejde či se nerozloučí, věnují mu pozornost neustále. */
    attentionSpan = nil

    /* Společná databáze odpovědí pro obě postavy. */
    proxyTarget = commonTopics
;

++ captainPaperWork: ConversationReadyState
    stateDesc = "Probírá se papíry. "
    specialDesc = "Kapitán s velitelem se probírají papíry. "
;

/*
 *   Jestliže mám v EventListu dva seznamy, první se nejprve jednou probere
 *   v pořadí definice a pak se teprve dělá běžný ShuffledEventList se druhým
 *   seznamem.
 */
+++ HelloTopic, ShuffledEventList
    [
        '<.p><q>Ehm,</q> odkašlal sis, <q>omlouvám se, že vás ruším.</q>

            <.p><q>No, jsi tady, tak povídej, co potřebuješ,</q> vyzval tě
                kapitán. '
    ]
    [

        '<.p>Tvůj pozdrav je očividně nevhod -- oba pánové velmi zaujatě
            probírají papíry na stole: <q>Potřebuješ něco, Tobiáši?</q> zeptal
            se kapián, aniž by zvedl hlavu. ',

        '<.p><q>Zdravím,</q> zahalekal jsi docela oprskle.
            <.p>Oba pánové zvedli hlavu a kapitán kývnul tvým směrem: <q>Co se
            děje, mladej?</q> '
    ]

    /* Zlikvidujeme default object announcement, který je k ničemu. */
    handleTopic(fromActor, topic)
    {
        gTranscript.clearReports();
        inherited(fromActor, topic);
    }
;

+++ ByeTopic, ShuffledEventList
    [
        '<.p><q>Tak já zas jdu,</q> přidržel ses dveří, ale už ti nikdo nevěnuje
            pozornost, pánové mají zajímavější téma, než jsou tvoje problémy. ',

        '<.p><q>Zatím,</q> rozloučil ses a oba pánové ti kývli v odpověď.
            Očividně vítají, že bys mohl konečně odejít. ',

        '<.p><q>Tak já jdu,</q> oznámil jsi.

            <.p><q>A koukej něco dělat,</q> zareagoval na tvé rozloučení kapitán
            a už tě nevnímá, obrátil se k veliteli. '
    ]
;

/* Pár témat se liší podle toho, koho jsem ze dvou postav primárně oslovil. */
++ AskTellTopic @suspiciousContainer
    "<.p>Nadechl ses ke vznesení dotazu, ale na poslední chvíli ses zarazil. Co
        když jeden z těch u bedny byl opravdu kapitán? A ty budeš dráždit
        chřestýše bosou nohou? Raději zkus jiný zdroj informací. "

    /*
     *   Protože tohle vlastně není konverzace, hráč si rozmyslí a nepoloží
     *   otázku, tak musíme označit, aby neproběhlo automatické pozdravení.
     */
    isConversational = nil
;

++ AskTellTopic @tChriss
    "<.p><q>Pane,</q> zeptal ses kapitána, <q>jaký byl vlastně Chriss?</q>

        <.p>Kapitán Henry pokrčil rameny, promnul si nos: <q>Těžko říci, létali
        jsme spolu dlouho, on byl takovej puntičkář, až šťoura, ve všem chtěl
        mít pořádek a jasno, až to hraničilo s posedlostí vědět o všem
        všechno.</q> "
;

/* ------------------------------------------------------------------------ */
/*
 *   V JÍDELNĚ
 */
+ captainInCanteenTalking: InConversationState
    stateDesc = "Mezi jednotlivými sousty se baví tak se všemi. "
    specialDesc = "Kapitán s velitelem se s tebou baví o opravě lodi. "
    attentionSpan = nil

    /*
     *   Po skončení rozhovoru se nevracíme do původního stavu, ale jdeme na
     *   natrpělivého kapitána.
     */
    nextState = commanderImpatient
;

++ captainInCanteen: ConversationReadyState
    stateDesc = "Sedí u stolu a plně se věnuje svému jídlu. Nevypadá, že by měl
        něco jiného než ostatní tady u stolu. "

    /* Potlačíme popis stavu při vejití do jídelny. */
    specialDesc = fixedComponents.location != nil ? "Kapitán s velitelem sedí u
        stolu a jedí. " : ""
;

+++ HelloTopic
    "<.p><q>Ještě dotázek,</q> ozval ses.

        <.p><q>Copak?</q> zamručel velitel. "
;

+++ ByeTopic, StopEventList
    [
        '<.p>Zvedl ses od stolu a odšoupl zbytky směrem k odpadu: <q>Děkuji za
            oběd, ty jahody byly fakt úžasné. A hlavně dík za součástky, jdu se
            pustit do práce, ať vám tu neužíráme kyslík příliš dlouho.</q>

            <.p><q>To jsem rád, že ti u nás chutnalo,</q> zareagoval velitel a
            kapitán dodal: <q>Zůstaneme tady v jídelně, přijď říct hned, jak
            budeš hotový, a vyrazíme zpátky. Ať tu nezdržujeme.</q> ',

        '<.p><q>Tak já jdu na to,</q> stručně jsi konstatoval.

            <.p><q>Ouki, a hlavně co nejdřív přijď oznámit, že máš hotovo, ať
            můžem letět,</q> poslal za tebou do dveří kapitán. '
    ]
;

/*
 *   Zde máme specifické DefaultTopicy kvůli formulaci s jídlem. To ale znamená,
 *   že zaujmou roli každého tématu, které zde nenaleznou a tím by překryla i
 *   všechna témata, které nejsou zde v captainInCanteenTalking, což nechceme.
 *   Proto pomocí deferToEntry řekneme, že jiná, než DefaultTopic témata se mají
 *   hledat v nadřazené databázi témat, tj. ne zde ve stavu postavy, ale přímo u
 *   postavy.
 */
++ DefaultAskTopic, ShuffledEventList
    [
        '<.p><q>Pane, mohu se zeptat…?</q> vznesl jsi dotaz. Kapitán nejdříve
            polknul a teprve pak dal na vědomí, že tvůj dotaz slyšel:
            <.p><q>Hmm, raději jez a nestarej se o hlouposti.</q> ',
        '<.p>Položil jsi otázku a upřel pohled na kapitána.
            <.p><q>Co bys asi myslel,</q> kapitán zvolna zpracovává sousto a
            nechává tě čekat, <q>že na tohle budu odpovídat?</q> ',
        '<.p><q>Jo, a ještě by mě zajímalo,</q> chtěl jsi pokračovat, ale
            velitel tě přerušil:
            <.p><q>Za našeho mládí byla jiná výchova,</q> a obrátil se ke
            kapitánovi, <q>holobrádci u stolu nesměli ani ceknout a nechávali
            lidi v klidu najíst.</q> '
    ]
    deferToEntry(entry) { return !entry.ofKind(DefaultTopic); }
;

++ DefaultAnyTopic
    "<.p><q>Pane,</q> pokusil ses upoutat pozornost svým problémem.
        <.p><q>Co má být?</q> zareagoval mezi dvěma sousty kapitán, <q>moji chuť
        k jídlu to nezmění.</q> "

    deferToEntry(entry) { return !entry.ofKind(DefaultTopic); }
;

/* ------------------------------------------------------------------------ */

++ ConvNode 'fixed-components'

    /* Možnost zeptat se na součásty musí zůstat, dokud neodejdeme. */
    isSticky = true

    /* Zde se iniciativa obrací, postava osloví hráče. */
    npcGreetingMsg()
    {
        "<.p><q>Hele, mladej,</q> oslovil tě kapitán, <q>tady máš ty součástky.
            A koukej, ať je to hotové ještě dneska.</q>

            <.p>Rozpačitě jsi natáhl ruku a převzal náhradní díly do vadného
                modulu.<.topics> ";

        /* Součástky jsou PresentLater, tak je zviditelníme. */
        fixedComponents.makePresent();
        fixedComponents.moveInto(me);
    }
;

+++ HelloTopic "Už spolu komunikujete. ";

+++ AskTellTopic, SuggestedAskTopic, StopEventList @fixedComponents
    [
        '<.p><q>To je úžasné!</q> zajásal jsi, <q>kde jste k nim
            přišli? <<if gRevealed('look-for-room')>>Boris mě už připravoval na
            několikaměsíční čekání.<<else>>Už jsem myslel, že tu na několik
            měsíců ztvrdneme.<<end>></q>

            <.p><q><<if gRevealed('look-for-room')>>To je technik na
            baterky,<<else>>To v nás nemáš trochu důvěry?<<end>></q> ohrnul nos
            velitel, <q>kdo ví a umí, dělá velitele.</q> A plácl si s kapitánem
            do dlaně.<.convstay><.topics> ',
        '<.p><q><<if gRevealed('look-for-room')>>Boris říkal<<else>>Myslel
            jsem<<end>>, že tahle série šla jenom do cestovních raket, takže
            se tady na základně v ničem nevyskytuje,</q> bráníš <<if
            gRevealed('look-for-room')>>kolegu<<else>>se<<end>>.

            <.p><q>Ale vůbec
            neví<<if gRevealed('look-for-room')>><<else>>š<<end>>,</q> oponuje
            velitel, <q>že se dají vytáhnout z havarovaných rypadel CMX12.</q>
            <<gSetKnown(tCmx12)>>

            <.p><q>Hmm,</q> říkáš si už v duchu, <q>k čemu by takové rypadlo
            potřebovalo součástku do výkonového vysílače?</q> '
    ]
    timesToSuggest = 2

    name = 'náhradní součástky'
;


/* ------------------------------------------------------------------------ */
/*
 *   VELITEL JE V JÍDELNĚ A JE NETRPĚLIVÝ
 */
+ commanderImpatientTalking: InConversationState
    stateDesc = "Je zvědavý, jak jsi pokročil s opravou. "
    specialDesc = "Velitel je zvědavý, jak jsi pokročil s opravou. "
;

++ commanderImpatient: ConversationReadyState
    stateDesc = "Sedí u stolu a s chutí jí. "
    specialDesc = "Kapitán s velitelem sedí u stolu a baví se u jídla. "

    afterTravel(traveler, connector)
    {
        captain.initiateConversation(commanderImpatientTalking, 'ship-fixed');
    }
;

+++ ByeTopic, ShuffledEventList
    [
        '<.p><q>Tak teda jdu pokračovat,</q> hodil jsi s nepatrným znechucením
            poznámku směrem ke kapitánovi.

            <.p>Jeho stejně lehce podrážděný tón se dal čekat: <q>Nemusíme tu
            čekat na příští zatmění Slunce, pohni sebou.</q> ',

        '<.p><q>Dobře, šéfe, jdu to dorazit,</q> pohodil jsi hlavou směrem ke
            kapitánovi.

            <.p><q>Hmm, tak ať ti to netrvá celou galaktickou hodinu,</q>
            zamručel ti v odpověď. ',

        '<.p><q>Nojo, co nadělám, jdu na tom máknout,</q> povzdechl sis okázale,
            aby si kapitán všiml, jaká jsi chudinka.

            <.p><q>Ono tě neubude,</q> zvedl k tobě hlavu kapitán, <q>a už vůbec
            ne, když trochu zakmitáš, molekulo.</q> '
    ]
;

++ DefaultAskTopic, ShuffledEventList
    [
        '<.p><q>Pane, mohu se zeptat…?</q> vznesl jsi dotaz. Kapitán nejdříve
            polknul a teprve pak dal na vědomí, že tvůj dotaz slyšel:
            <.p><q>Hmm, raději jez a nestarej se o hlouposti.</q> ',
        '<.p>Položil jsi otázku a upřel pohled na kapitána.
            <.p><q>Co bys asi myslel,</q> kapitán zvolna zpracovává sousto a
            nechává tě čekat, <q>že na tohle budu odpovídat?</q> ',
        '<.p><q>Jo, a ještě by mě zajímalo,</q> chtěl jsi pokračovat, ale
            velitel tě přerušil:
            <.p><q>Za našeho mládí byla jiná výchova,</q> a obrátil se ke
            kapitánovi, <q>holobrádci u stolu nesměli ani ceknout a nechávali
            lidi v klidu najíst.</q> '
    ]

    deferToEntry(entry) { return !entry.ofKind(DefaultTopic); }
;

++ DefaultAnyTopic
    "<.p><q>Pane,</q> pokusil ses upoutat pozornost svým problémem.
        <.p><q>Co má být?</q> zareagoval mezi dvěma sousty kapitán, <q>moji chuť
        k jídlu to nezmění.</q> "

    deferToEntry(entry) { return !entry.ofKind(DefaultTopic); }
;

/* ------------------------------------------------------------------------ */

+ ConvNode 'ship-fixed'

    /* Kapitán je netrpělivý odletět, tak furt otravuje. */
    npcGreetingMsg = "<.p>Kapitán při tvém příchodu okamžitě přerušil rozhovor s
        velitelem a obrátil se k tobě: <q>Hotovo? Můžeme vyrazit?</q><.topics> "

    npcContinueMsg = "<q>A ještě mi řekni,</q> ozval se kapitán, <q>je už loď
        opravená?</q> "
;

/* Není suggested, ale pro jistotu máme odpověď. */
++ YesTopic
    "<.p>To bys asi neměl říkat, když to není pravda. To je jako chodit při
        bouřce s vidlema a pak se divit, co že se to stalo. "

    isConversational = nil
;

+++ AltTopic, SuggestedYesTopic
    "<.p><q>Jasně, všechno je na svém místě, odpovídáš.</q> Všímáš si letmého
        pohledu plného uspokojení, který si vyměnili oba šéfové a najednou
        váháš.

        <.p>Je tu hromada nezodpovězených otázek. Kde se vzaly ty náhradní
        součástky? Co je v té zakázané oblasti? Co tam jeli kontrolovat šéfové,
        když je těžba dávno odkloněná a oblast uzavřená? A co ta divná bedna?

        <.p><q>Tak můžeme startovat?</q> důraz v kapitánově hlase tě vytrhnul z
        úvah.<.convnode ship-fixed-again><.topics> "

    timesToSuggest = nil
    isActive = gRevealed('ship-fixed')
;

++ NoTopic, SuggestedNoTopic
    "<.p>Najednou ti připadá ten spěch podezřelý: <q>Jo, modul už je na svém
        místě,</q> informuješ kapitána, <q>ale odletět ještě nemůžeme. Spustil
        jsem testovací programy a potřebuju dodělat kalibrace,</q> doufáš, že se
        ti podařilo rafinovanou výmluvou získat víc času.

        <.p><q>Dobrá tedy,</q> odvětil kapitán, <q>ale pospěš si.</q> "

    timesToSuggest = nil
;

+++ AltTopic, SuggestedNoTopic
    "<.p><q>Ne,</q> odpověděl jsi, <q>ještě to nemám. Nejde to zas tak
        rychle.</q>

        <.p><q>Tak se tu necourej,</q> doporučil ti kapitán, <q>a vrať se k
        práci.</q> "

    timesToSuggest = nil
    isActive = !gRevealed('ship-fixed')
;

/* ------------------------------------------------------------------------ */

+ ConvNode 'ship-fixed-again'
    npcContinueMsg = "<q>A ještě mi řekni,</q> ozval se kapitán, <q>je už loď
        opravená?</q> "
;

++ YesTopic, SuggestedYesTopic
    topicResponse()
    {
        "<.p><q>Ano, loď je připravena ke startu,</q> kývl jsi kapitánovi na
            souhlas, můžeme odletět hned, jak se nalodíme.

            <.p>Rozloučili jste se s velitelem, vzkázali pozdrav Borisovi a po
            krátké předletové přípravě jste nabrali kurz domů. Zanedlouho ti
            kapitán předal službu a ty jsi zůstal v kabině sám jen se svými
            myšlenkami. Je zajímavé, že necítíš pocit uspokojení, že se vracíte
            v pořádku zpátky. ";

/* Když hrály děti soutěžně, byl na webu žebříček v reálném čase. */
//#ifdef TADS_INCLUDE_NET
//        local data =
//        [
//            'sid' -> webSession.storageSID,
//            'event' -> 'surrender1',
//            'points' -> 0,
//            'desc' -> '',
//            'total' -> libScore.totalScore
//        ];

//        sendNetRequest(0, 'http://technoplaneta.cz/t3submit', 'POST', 0, nil,
//            data);
//#endif

        /* Jeden z neúspěšných konců. */
        finishGameMsg('VZDAL SES', [ finishOptionUndo, finishOptionCredits,
            finishOptionFullScore ]);

    }

    isActive = gRevealed('ship-fixed')
;

++ NoTopic, SuggestedNoTopic
    "<.p><q>Jo, modul už je na svém místě,</q> informuješ kapitána, <q>ale
        odletět ještě nemůžeme. Spustil jsem testovací programy a potřebuju
        dodělat kalibrace,</q> doufáš, že se ti podařilo rafinovanou výmluvou
        získat víc času.

        <.p><q>Dobrá tedy,</q> odvětil kapitán, <q>ale pospěš si.</q> "
;

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN S VELITELEM PRONÁSLEDUJÍ HRÁČE
 */
+ captainChasing: HermitActorState
    stateDesc = "Stojí za velitelem a vypadá odhodlaně. "

    /*
     *   Necháme úmyslně prázdné a přitom zadané, aby se negeneroval implicitní
     *   group listing.
     */
    specialDesc = ""
    noResponse = "Kapitán se ve vysílačce nehlásí, spojení je přerušené. "
;
