#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>
#include <date.h>

/* ------------------------------------------------------------------------ */
/*
 *   Třída pro oblečení, které má postava na sobě už od začátku hry. Oblečení
 *   je potřeba postavě dát pomocí syntaxe s plusem.
 */
class InitiallyWorn: Wearable
    wornBy = (location)

    /*
     *   Takové objekty schováme před VŠE, protože jsou to přirozené součásti
     *   osoby, která je nosí.
     */
    hideFromAll(action) { return true; }
;

/*
 *   Třída pro oblečení postavy hráče, které si nemůže sundat. Používá se pro
 *   běžné oblečení hráče, které není důvod sundavat.
 */
class AlwaysWorn: Wearable
    wornBy = me
    dobjFor(Doff)
    {
        check()
        {
            cannotDoffMsg;
            exit;
        }
    }
    cannotDoffMsg = "Teď opravdu není důvod, proč si něco sundavat. "

    dobjFor(Drop) asDobjFor(Doff)

    /* Schováme před "polož vše" či "svlékni vše". */
    hideFromAll(action)
    {
        return action.ofKind(DropAction)
            || action.ofKind(DoffAction)
            || action.ofKind(PutInAction)
            || inherited(action);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Místnosti, které postava hráče navštíví, označujeme vlajkou visited. To se
 *   hodí při zobrazování a skrývání rzných nápověd či tipů.
 */
modify Room
    visited = nil

    travelerArriving(traveler, origin, connector, backConnector)
    {
        inherited(traveler, origin, connector, backConnector);
        if(me.isOrIsIn(traveler))
            visited = true;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Modifikace třídy Thing, aby uměla pomocí metody listRemoteContents() vypsat
 *   obsah předmětů ve vzdálené místnosti, do které hráč nahlíží.
 */
modify Thing
    listLocation_ = nil
    listRemoteContents(otherLocation)
    {
        listLocation_ = otherLocation;
        try
        {
            lookAround(gActor, LookListPortables);
        }
        finally
        {
            listLocation_ = nil;
        }
    }

    adjustLookAroundTable(tab, pov, actor)
    {
        inherited(tab, pov, actor);
        if(listLocation_ !=  nil)
        {
            local lst = tab.keysToList();
            foreach(local cur in lst)
            {
                if(!cur.isIn(listLocation_))
                    tab.removeElement(cur);
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Speciální verze ukecaného konektoru, jehož ukecanost se dá vypnout a
 *   zapnout. Používáme např. v přechodovém tunelu do lodi, kde zprávu o
 *   průchodu vypneme při úvodním vystoupení z lodi, protože to hráče provází
 *   kapitán a máme už výstup ohlášený jinak.
 */
class SwitchableTravelWithMessage: TravelWithMessage
    enabled = true

    enable() { enabled = true; }
    disable() { enabled = nil; }

    showTravelDesc() { if(enabled) inherited; }
;

/* ------------------------------------------------------------------------ */
/*
 *   Tato agenda zajišťuje přesun NPC postavy po stanovené trase. Používáme
 *   např. když kapitán s velitelem jdou z kupole do kajuty.
 */
class TravelAgendaItem: AgendaItem
    travelPath = nil
    invokeItem()
    {
        local actor = getActor;
        local path = nilToList(travelPath);
        local loc = actor.getOutermostRoom();
        local idx = path.indexOf(loc);

        if(idx && idx < path.length())
        {
            if(me.canSee(loc)) me.trackFollowInfo(actor, path[idx + 1], loc);
            actor.scriptedTravelTo(path[++idx]);
        }

        if(idx == nil || idx >= path.length())
        {
            isDone = true;
            destReached();
        }
    }
    destReached() { }
;

/* ------------------------------------------------------------------------ */
/*
 *   "Disambiguation deferrer" is a mix-in class that can be combined with
 *   any ordinary Thing class to create an object that defers in
 *   disambiguation situations to a given enumerated set of objects, or to
 *   objects that match some rule.  If we find any object from our
 *   enumerated set, or any object that matches our rule, we'll take
 *   ourselves out of the picture for disambiguation purposes.
 *
 *   Deferral is useful in a lot of cases.  Frequently, one object will
 *   have a much stronger affinity for a given set of vocabulary words than
 *   most other objects, but we'll nonetheless want to make the other
 *   objects match the vocabulary when the stronger object isn't around.
 *   The deferrer lets the weaker objects opt out when they see the
 *   stronger object as a possible match.
 */
class DisambigDeferrer: object
    /*
     *   A list of objects I defer to.  We'll remove ourselves from parsing
     *   consideration if any of these objects are in scope.
     */
    disambigDeferTo = []

    /*
     *   Do we defer to the given object?  Returns true if we defer to the
     *   given object, nil if not.  By default, we defer if the object
     *   appears in our disambigDeferTo list.
     */
    disambigDeferToObj(obj) { return disambigDeferTo.indexOf(obj) != nil; }

    /*
     *   Filter the noun-phrase resolution list.  If we find any objects
     *   that we defer to in the match list, we'll defer by removing 'self'
     *   from the match list.
     */
    filterResolveList(lst, action, whichObj, np, requiredNum)
    {
        /* if there's an object we defer to, defer to it */
        if (lst.indexWhich({x: disambigDeferToObj(x.obj_)}) != nil)
        {
            /* find myself in the list */
            local i = lst.indexWhich({x: x.obj_ == self});

            /* remove myself from the list */
            lst = lst.removeElementAt(i);
        }

        /* return the updated list */
        return lst;
    }
;

/*
 *   Component Deferrer - this is a disambiguation deferrer that can be
 *   combined with any component class to make the component defer to any
 *   non-component object.  Components are usually minor parts added to
 *   enhance the simulation depth, but as minor parts, they shouldn't take
 *   precedence over any more important objects in the game.
 */
class ComponentDeferrer: DisambigDeferrer
    /* defer to any non-component */
    disambigDeferToObj(obj) { return !obj.ofKind(Component); }
;

/* ------------------------------------------------------------------------ */
/*
 *   Zde řešíme problém s pořadím vykonávání jednotlivých věcí, vynutíme
 *   zobrazení jednoho tipu, o ktery bychom jinak přišli.
 */
modify SuggestedTopicLister
    showListSuffixWide(cnt, pov, parent, selector)
    {
        inherited(cnt, pov, parent, selector);
        if(me.isIn(supplyRoom))
        {
            gReveal('hint-topics');
            extraHintManager.hintDaemon();
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Když hráč neřekne, v čem chce hledat, tak se hra napoprvé zeptá, ale pak
 *   se už neptá a automaticky hledá ve stejném objektu, což je v praxi tablet
 *   a jeho Spacepedie. Ve skladu to ale může zastínit možnost hledat v regálech
 *   a v podpalubí může nevinné "hledej zdroj světla" prozradit řešení puzzlu.
 *   Proto v těchto situacích chceme, aby hráč vždy řekl, v čem hledá. Toto
 *   zařídí, aby nehledal ve stejném objektu, kde hledal naposledy a ještě
 *   musíme tablet učinit nonObvious.
 */
modify ConsultAboutAction
    getDefaultDobj(np, resolver)
    {
        if(me.isIn(lowerDeck) && lowerDeck.scoreMarker.scoreCount == 0
            || me.isIn(supplyRoom) && gRevealed('low-stock') &&
            !gRevealed('viz54-found'))
            return inherited TAction.getDefaultDobj(np, resolver);
        else
            return inherited(np, resolver);
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Když hrály děti soutěžně, tak rady k příběhu byly načasované, aby se
 *   objevily až po týdnu.
 */
modify hintManager
    showHints()
    {
        /* if there is no top-level hint menu, no hints are available */
        local now = new Date();
        local hint = new Date(2013, 4, 29, 6, 0, 0, 0);
        if(now < hint)
        {
            mainReport('Rady pro příběh budou na tomto místě dostupné až 29.
                dubna v 6.00 hod. ');
            return;
        }

        inherited;
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Ve hře máme několik DistanceConnectorů, které propojují místnosti v lodi a
 *   na základně. Na dálku chceme ale vidět jen postavy, ne předměty, proto zde
 *   globálně snížíme jejich velikost.
 */
modify Thing
    sightSize = small
;

modify Actor
    sightSize = medium
;

modify BasicLocation
    sightSize = medium
;

modify TravelConnector
    sightSize = medium
;


/* ------------------------------------------------------------------------ */
/*
 *   Postavu, která nás provází, označíme za zvláště vhodnou pro následování.
 */
modify TourGuide
    dobjFor(Follow)
    {
        verify()
        {
            /*
             *   If the actor can see us, and we're in a "guided tour"
             *   state, we can definitely perform the travel.  Otherwise,
             *   use the standard "follow" behavior.
             */
            if (gActor.canSee(self) && getTourDest() != nil)
            {
                /*
                 *   we're waiting to show the actor to the next stop on
                 *   the tour, so we can definitely proceed with this
                 *   action
                 */
                logicalRank(110, 'tour-guide');
            }
            else
            {
                /* we're not in a tour state, so use the standard handling */
                inherited();
            }
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Když je objekt Wearable, tak má dobjFor(Remove) asDobjFor(Doff) a to se
 *   vůbec nehodí pro objekt, který nemá postava na sobě, jako karta, kterou
 *   může hráč dát do slotu a pak říci "vyndej kartu".
 */
modify Wearable
    dobjFor(Remove)
    {
        preCond = [touchObj, objNotWorn, roomToHoldObj]
        verify()
        {
            /* if we're already held, there's nothing to remove me from */
            if (isHeldBy(gActor))
                illogicalNow(&cannotRemoveHeldMsg);
        }
        action() { askForIobj(TakeFrom); }
    }

    tryWearing()
    {
        /*
         *   Try an implicit 'take' command.  If the actor is carrying the
         *   object indirectly, make the command "take from" instead,
         *   since what we really want to do is take the object out of its
         *   container.
         */
        if (!isWornBy(gActor))
            return tryImplicitAction(Wear, self);
        else
            return inherited();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Kafovaky v zásobníku jsou stejně důležité, jako ty mimo něj.
 */
modify Dispensable
    dobjFor(All)
    {
        verify() {}
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Tip o otázce parseru, co má hráč na mysli, nedokážeme zařídit pomocí výše
 *   uvedeného mechanismu a tak ho přidáme mimo.
 */
modify playerMessages
    askDisambig(actor, originalText, matchList, fullMatchList,
            requiredNum, askingAgain, dist)
    {
        inherited(actor, originalText, matchList, fullMatchList, requiredNum,
            askingAgain, dist);

        if(!gRevealed('disambig'))
        {
            "<.p><font color=\"#808080\"><b>Tip:</b> Když se tě příběh zeptá
                <q>A co tím máš na mysli?</q>, tak stačí odpovědět např. jen
                přídavným jménem či dokonce jen pořadovou číslovkou (např.
                PRVNÍ). Nemusíš opravovat a zadávat celý příkaz znovu.</font> ";

            gReveal('disambig');
        }
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Hráči neznalý textových her by mohli mít tendenci zkoušet zadat příkaz
 *   "oprav". U objektů, které potřebují opravit, pak reagujeme vysvětlením, že
 *   hráč musí popsat konkrétně, jak chce objekt opravit.
 */
DefineTAction(Repair);

VerbRule(Repair) ('oprav' | 'opravit' | 'sprav' | 'spravit') singleDobj
    : RepairAction
    verbPhrase = 'opravit/opravu{ješ}/opravil{a} (co)'
;

modify Thing
    dobjFor(Test)
    {
        verify() { logicalRank(50, 'not testable'); }
        action() { askForIobj(TestWith); }
    }

    dobjFor(TestWith)
    {
        verify() { logicalRank(50, 'not testable'); }
    }
    iobjFor(TestWith)
    {
        verify() { illogical('Pomocí {kohočeho iobj} nemůž{eš} nic
            otestovat. '); }
    }

    dobjFor(Repair)
    {
        preCond = [touchObj]
        verify() { }
        action() { "Budeš muset říci přesněji, jak to chceš udělat. "; }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   No a nakonec co v žádné správné textovce nesmí už od dob Adventure chybět.
 */
DefineIAction(XYZZY)
    execAction()
    {
        response.doScript();
    }

    response: StopEventList
    {[
        '<.p>Úplně se ti tím kouzlem zatočila hlava a na chvíli jsi ztratil
            orientaci: <q>Sever, jih, západ, východ, nahoru, dolů</q> zopakoval
            sis několikrát, než ses opět bezpečně zorientoval v ortogonálním
            světě. '
    ]}

    actionTime = 0
;

VerbRule(XYZZY)
    ('řekni' | 'říci' | 'vyslov' | 'vyslovit' | ) 'xyzzy'
    : XYZZYAction
    verbPhrase = 'vyslovit/vyslovu{ješ}/vyslovil{a} XYZZY'
;
