#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/*
 *   Většina logiky připojování a testování je převzata z Return to Ditch Day,
 *   kde byly podobné přístroje.
 */

/* we might want to test things with the tester in the introduction */
DefineTAction(Test);
VerbRule(Test)
    ('testuj' | 'testovat' | 'otestuj' | 'otestovat' | 'vyzkoušej' | 'vyskoušet'
    | 'diagnostikuj' | 'diagnostikovat') singleDobj : TestAction
    verbPhrase = 'otestovat/testu{ješ}/otestoval{a} (co)'
;

DefineTIAction(TestWith);
VerbRule(TestWith)
    ('testuj' | 'testovat' | 'otestuj' | 'otestovat' | 'vyzkoušej' | 'vyskoušet'
    | 'diagnostikuj' | 'diagnostikovat') singleDobj singleIobj : TestWithAction
    verbPhrase = 'otestovat/testu{ješ}/otestoval{a} (co) (čím)'
;

/*
 *   A class for the testing gear (oscilloscope, signal generator).  These
 *   objects can be attached to testable equipment; they have to be
 *   located right next to the equipment being tested (i.e., in the same
 *   container), which we enforce by making these NearbyAttachable
 *   objects.
 */
class TestGear: PlugAttachable, NearbyAttachable
    /* we can attach to anything that's a test-gear-attachable object */
    canAttachTo(obj)
    {
        return obj.ofKind(TestGearAttachable);
    }
    explainCannotAttachTo(other)
    {
        gMessageParams(self, other);
        "Nevypadá to, že by {kdoco other} měl vhodný konektor, kam by se dal
            {kdoco self} připojit. ";
    }

    /*
     *   We can only be attached to one thing at once, so add a
     *   precondition to ATTACH TO requiring that we're not attached to
     *   anything.
     */
    dobjFor(AttachTo) { preCond = (inherited() + objNotAttached) }
    iobjFor(AttachTo) { preCond = (inherited() + objNotAttached) }

    /* for electrical safety, turn off the test equipment before unplugging */
    dobjFor(Detach) { preCond = (inherited() + objTurnedOff) }
    dobjFor(DetachFrom) { preCond = (inherited() + objTurnedOff) }
    iobjFor(DetachFrom) { preCond = (inherited() + objTurnedOff) }

    /* TEST obj WITH gear is the same as ATTACH gear TO obj */
    iobjFor(TestWith) remapTo(AttachTo, self, DirectObject)

    /*
     *   Get the equipment we're attached to.  We can only be attached to
     *   one thing at a time, other than our permanent attachments, so this
     *   returns either nil or the single piece of equipment we're probing.
     */
    getAttachedEquipment()
    {
        local lst;

        /* get the list of non-permanent attachments */
        lst = attachedObjects.subset({x: !isPermanentlyAttachedTo(x)});

        /*
         *   there can be no more than one, so if we have anything in our
         *   list, return the first one; if not, return nil
         */
        return (lst.length() != 0 ? lst[1] : nil);
    }

    /* show what happens when we probe with the oscilloscope */
    probeWithUnidip() { "The oscilloscope screen just shows a flat line. "; }
;

/*
 *   Our test gear tends to have attached connectors.  We include these for
 *   completeness; they really don't do much apart from reflect commands
 *   back to the test gear container.
 */
class PluggableComponent: PlugAttachable, Component
    /* attaching the probe really attaches the containing test gear */
    dobjFor(AttachTo) remapTo(AttachTo, location, IndirectObject)
    iobjFor(AttachTo) remapTo(AttachTo, DirectObject, location)

    /* defer to the parent for status, for the attachment list */
    examineStatus() { location.examineStatus(); }
;

/*
 *   a probe for a piece of test gear - this is a test gear component
 *   that's also a permanent attachment child
 */
class TestGearProbe: PermanentAttachmentChild, PluggableComponent
    /* putting the probe on something is equivalent to attaching it */
    dobjFor(PutOn) remapTo(AttachTo, location, IndirectObject)
;

/*
 *   A class representing objects that can attach to the testing gear
 *   (such as the oscilloscope and the signal generator).  These are all
 *   NearbyAttachments, because the test gear has to be placed right next
 *   to them (i.e., in the same container) to be attached.  Apart from
 *   making the object attachable, this is essentially just a marker class
 *   that we recognize in the test gear, to make sure that the object is
 *   something the scope can attach to.
 */
class TestGearAttachable: PlugAttachable, NearbyAttachable
    canAttachTo(obj)
    {
        return obj.ofKind(TestGear);
    }
    
    /* show what happens when we probe with the oscilloscope */
    probeWithUnidip() { }

    /* show what happens when we attach the signal generator */
    probeWithSignalGen() { }

    /* testing this kind of thing is likely */
    dobjFor(Test) { verify() { } }

    /*
     *   These objects are always the "major" items when attached to test
     *   gear: "the oscilloscope is attached to the box," not vice versa.
     */
    isMajorItemFor(obj) { return obj.ofKind(TestGear); }
;

/* a transient object to keep track of our scope notes per session */
transient unidipNoteTracker: object
    noteCount = 0
;

/*
 *   UniDiP, předchůdce slavného VUniDiPu, je náš diagnostický přístroj, který
 *   pomůže hráči v odhalení závady na lodi.
 */
unidip: TestGear, OnOffControl, CustomFixture 
    'diagnostický přístroj/unidip' 'UniDiP' *2 @srTable
    "Univerzální diagnostický přístroj umí zjistit nejrůznější závady běžně
        používaných částí techniky používané v kosmu. Vede od něj kabel se
        sondou, která se připojuje k testovaným obvodům. <<screenDesc>> "

    cannotTakeMsg = 'UniDiP nemůžeš odnést, je připevněný a vede od něho spousta
        kabelů, které nechceš rozpojovat. '
    cannotAttachMsg = 'Toto není předmět testovatelný UniDiPem. '
    cannotAttachToMsg = 'Toto není předmět testovatelný UniDiPem. '
    cannotAttachToSelfMsg = 'Nemáš více UniDiPů, abys mohl UniDiPem zkoumat
        UniDip. '

    /* make it a precondition of attaching that we're turned on */
    dobjFor(AttachTo) { preCond = (inherited() + objTurnedOn) }
    iobjFor(AttachTo) { preCond = (inherited() + objTurnedOn) }

    /* handle attaching to another object */
    handleAttach(other)
    {
        /* let the other object tell us what happens */
        other.probeWithUnidip();

        /*
         *   We don't actually stay attached; attaching to us is just
         *   momentary probing.  So mark us as no longer attached.
         */
        detachFrom(other);
    }

    screenDesc()
    {
        if (!isOn)
            "UniDiP je momentálně vypnutý. ";
        else
            "UniDiP je zapnutý, ale čeká na připojení elektronického zařízení,
                které má být otestováno. Na obrazovce zobrazuje jen nudné
                rotující logo společnosti TechnoUniverse Corp. ";
    }

    dobjFor(TurnOn)
    {
        action()
        {
            /*
             *   announce this only if we're an explicit action; turning
             *   the scope on is a pretty simple action, so it's not worth
             *   a mention when it's done incidentally to another action
             */
            if (!gAction.isImplicit)
            {
                "Zapnul jsi UniDiP. Kontrolky spokojeně zablikaly a zahrála
                    krátká fanfára. Zdá se, že je připraven k použití. ";

                if (timesOn++ > 0)
                    "Opět jsi zapnul UniDiP. Zapínací znělka už začíná být
                        trochu otravná. Nechtěl bys jím raději něco
                        otestovat? ";
            }

            /* mark it as on */
            makeOn(true);
        }
    }

    /* number of times we've turned it on */
    timesOn = 0

    dobjFor(TurnOff)
    {
        action()
        {
            /* do the normal work */
            inherited();

            /*
             *   if we're not running as an implied action, mention that
             *   the screen goes dark
             */
            if (!gAction.isImplicit)
                "UniDiP se po stisknutí tlačítka chvíli ukončoval, pak smutně
                    zapípal a vypnul se. ";
        }
    }

    iobjFor(AimAt)
    {
        action()
        {
            "Na obrazovce tabletu se zobrazila pouze krátká informace.

                <.p>UniDiP je univerzální diagnostický přístroj vyvinutý v
                laboratořích TechnoUniverse Corporation. Používejte obezřetně k
                hledání závad v elektronickém vybavení z našeho výrobního
                programu.

                <.p>Nepropadejte panice! ";
        }
    }

    gcName = 'UniDiPu, UniDiPu, UniDiP, UniDiPu, UniDiPem'
    gcVocab = 'diagnostický diagnostického diagnostickému diagnostickém
        diagnostickým přístroje/přístroji/přístrojem/unidipu/unidipem'
;

/*
 *   Our test gear tends to have attached connectors.  We include these for
 *   completeness; they really don't do much apart from reflect commands
 *   back to the test gear container.
 */
+ probe: ComponentDeferrer, TestGearProbe
    'sonda kabel unidipu/sonda/kabel' 'sonda' *3
    "K UniDiPu je připojená univerzální sonda, kterou znáš z dávných školení. Je
        vymyšlená tak, aby ji bylo možno připojit k libovolnému podezřelému
        elektronickému modulu. "

    changeGender = 'kabel:2'
    gcName = 'sondy, sondě, sondu, sondě, sondou'
    gcVocab = 'sondy/sondě/sondu/sondou/kabelu/kabelem'
;

+ ComponentDeferrer, Component
    'obrazovka (unidipu)/obrazovka' 'obrazovka UniDiPu' *3
    desc()
    {
        if (location.isOn)
            location.screenDesc();
        else
            "Obrazovka je zhasnutá. ";
    }

    gcName = 'obrazovky UniDiPu, obrazovce UniDiPu, obrazovku UniDiPu,
        obrazovce UniDiPu, obrazovkou UniDiPu'
    gcVocab = 'obrazovky obrazovce obrazovku obrazovkou obrazovky/obrazovce/
        obrazovku/obrazovkou'
;
