#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 */

#include <adv3.h>
#include <cs_cz.h>

/*
 *   Zavedeme slovník pro abstraktní témata, kterým neodpovídá žádný objekt ve
 *   hře, ale hráč se na ně může chtít vyptávat. Dáváme 1., 4. a 6. pád.
 */
tChriss: Topic 'chrissovu andersonovu nehodu nehoda chriss chrisse andersona
    chrisovu chris chrise -';
tIncident: Topic 'nehodu nehodě havárii havárku havárce rypadla rypadlo -';
tMining: Topic 'přesun odklon pokles těžby těžbu těžba -';

/*
 *   Definice postavy velitele. Základem je třída Person a doplňujeme TourGuide
 *   kvůli společnému vstupu do jídelny, kdy nás velitel vede, a zvláštní třídu
 *   ConversationStateSync, která zajišťuje, aby v jednu chvíli v kajutě
 *   velitele probíhal rozhovor s oběma postavami najednou, jako by byli oba
 *   jednou postavou.
 */
commander: DisambigDeferrer, ConversationStateSync, TourGuide, Person
    'ivan petronov/velitel' 'velitel' *1 @dome
    "Určitě je starší, pravděpodobně jako kapitán. Zdá se, že i v tomhle
        vesmírném zapadákově dbá na pořádek i disciplínu. Má přísný vzhled,
        vlasy ostříhané na ježka a čistý, pečlivě upravený pracovní overal. "

    /*
     *   Konfigurace třídy ConversationStateSync, která zajišťuje synchronizaci
     *   konverzačních stavů obou postav v kajutě.
     */
    otherActor = captain
    myConversationReadyState = commanderPaperWork
    myInConversationState = commanderPaperWorkTalking
    otherConversationReadyState = captainPaperWork
    otherInConversationState = captainPaperWorkTalking

    /*
     *   Během společného rozhovoru ve velitelově kajutě nechceme nutit hráče,
     *   aby si vybíral mezi velitelem a kapitánem, když si přijde promluvit.
     *   Proto velitele upozadíme.
     */
    disambigDeferToObj(obj)
        { return obj == captain && me.isIn(commandersRoom); }

    /*
     *   Při spojení s velitelem přes vysílačku musíme vyřadit z činnosti filtr,
     *   který by filtroval hlášky z npcContinueList kvůli tomu, že velitel není
     *   vidět. Při volání callWithSenseContext tedy uvedeme místo původního
     *   sight raději sound, jiné změny oproti knihovně zde nejsou.
     */
    executeTurn()
    {
        if(commander.curState == commanderOnRadio)
        {
            /* start a new command visually when a new actor is taking over */
            "<.commandsep>";

            /*
             *   Execute the turn in a daemon action context, and in the sight
             *   context of the actor.  The sense context will ensure that we
             *   report the results of the action only if the actor is visible
             *   to the player character; in most cases, the actor's
             *   visibility is equivalent to the visibility of the effects, so
             *   this provides a simple way of ensuring that the results of
             *   the action are reported if and only if they're visible to the
             *   player character.
             *
             *   Note that if we are the player character, don't use the sense
             *   context filtering -- we normally want full reports for
             *   everything the player character does.
             */
            return withActionEnv(EventAction, self,
                {: callWithSenseContext(isPlayerChar() ? nil : self, sound,
                                        {: executeActorTurn() }) });
        }

        return inherited;
    }

    /*
     *   Při rozhovoru přes vysílačku na konci hry potřebujeme, aby hráč ztratil
     *   klid a začal konfrontovat velitele s usvědčujícími důkazy, které viděl
     *   v trezoru. To je důležité, aby se situace vyhrotila v rozhodnutí
     *   velitele a kapitána, že hráče zabijí. Pokud se hráč ptá jen na
     *   neinkriminující věci, postupně ho tlačíme k větší konfrontaci.
     */
    disapointingCounter = 0
    disapointingAnswer()
    {
        /* Pár význačných hlášek v daném čase, jinak točíme dokola eventlist. */
        if(++disapointingCounter == 3)
        {
            "<.p>Tak tohle tě už přestává bavit. Na každou otázku jen vyhýbavé
                odpovědi, nedozvíš se nic a přitom cítíš každým kouskem těla, že
                něco není v pořádku. Musíš se dovědět něco víc. ";
        }
        else if(disapointingCounter == 5)
        {
            "<.p>Chceš přece vědět, co se tu děje a očividně ti to rovnou
                neřeknou. Tak nějaké téma více na tělo! ";
        }
        else
        {
            disapointingList.doScript();
        }
    }
    disapointingList: ShuffledEventList
    {
        [
            '<.p><q>To jsem se toho dozvěděl,</q> pomyslel sis. ',
            '<.p><q>Kdyby mi neřekl nic, tak se toho dozvím stejně,</q> pomyslel
                sis. ',
            '<.p><q>Nojo, víc bych se dověděl, kdybych se zeptal tkaniček u
                svých bot,</q> pomyslel sis. ',
            '<.p><q>Tohle je chození kolem horké kaše,</q> pomyslel sis, <q>s
                takovou tu můžem být do Vánoc.</q> '
        ]
    }

    /*
     *   Na konci hry je kapitán i velitel ve vozítku. Pokud hráč chce střílet
     *   na postavu, tak vlastně tím myslí vozítko.
     */
    dobjFor(ShootAt)
        maybeRemapTo(me.isIn(tunnel), ShootAt, secondRover, IndirectObject)
    dobjFor(Attack)
        maybeRemapTo(me.isIn(tunnel), Attack, secondRover)

    /*
     *   Ve slovníku jsou přivlastňovací přídavná jména postav (např. velitelův)
     *   označena podle anglické zvyklosti apostrofem s písmenkem s. Knihovna je
     *   podle toho rozpozná a pamatuje si, že jsou přivlastňovací.
     */
    gcName = 'velitele, veliteli, velitele, veliteli, velitelem'
    gcVocab = 'ivana ivanovi ivane ivanu ivanem petronova/petronove/petronovi/
        petronovem/velitele/veliteli/velitelovi/velitelem/velitelův\'s/
        velitelova\'s/velitelovo\'s'
;

/* ------------------------------------------------------------------------ */
/*
 *   Nejprve globání témata konverzace přidaná přímo do postavy. Ty jsou k
 *   dispozici vždy, pokud není řečeno jinak pomocí isActive.
 */
+ AskTellTopic @technician
    "<.p><q>Veliteli, mohu se vás zeptat?</q> kladeš otázku, <q>co je za
        člověka, ten váš technik?</q>

        <.p><q>Myslíš Borise?</q> obrátil se k tobě velitel, <q>odborník na svém
        místě, má znalosti i šikovné ruce, až na to jeho kafování -- bez pytlíku
        toho černýho vývaru nevydrží snad ani minutu.</q> "
;

+ AskTellTopic @captain
    "<.p><q>Veliteli?</q> vznesl jsi dotaz, <q>znáte kapitána Scotta už hodně
        dlouho?</q>

        <.p><q>Roky, chlapče, celé roky, ale nestalo se v nich nic, co by tě
        mohlo zajímat,</q> odpálkoval tě velitel. "
;

+ AskTellTopic @tresor
    "<.p><q>Jé,</q> překvapeně jsi zareagoval, <q>vy tu máte trezor?</q>

        <.p>Velitel pokrčil rameny: <q>Asi zlozvyk ze Země nebo nostalgie?</q> "
;

/*
 *   Pokud postava nemá odpověď, řekně nějakou z výmluv níže. Rozlišíme výmluvy
 *   při otázce a při ostatních konverzačních příkazech, kdy jsou trochu jinak
 *   formulované. ShuffledEventList po vyčerpání odpovědí je znovu zamíchá.
 *   Reveal default-topic zařídí zobrazení tipu tutoriálu.
 */
+ DefaultAskTopic, ShuffledEventList
    [
        '<.p>Pokusil ses oslovit velitele, ale ten zívá a vůbec tě nevnímá.
            <.reveal default-topic> ',
        '<.p><q>Veliteli, mohl bych…?</q> začal jsi mluvit, ale velitel tě
            otráveně přerušil:
            <.p><q>Co jsi říkal?</q><.reveal default-topic> ',
        '<.p>Nadechl ses: <q>Pane, rád bych…,</q> ale velitel ti nedal domluvit:
            <.p><q>Dej pokoj, nemusíš vědět všechno.</q>
            <.reveal default-topic> ',
        '<.p><q>Můžu se zeptat?</q> obrátil ses k veliteli.
            <.p><q>Kdo se moc ptá, moc se dozví,</q> reaguje na tvou otázku
            velitel.<.reveal default-topic> ',
        '<.p><q>Pane,</q> ale než jsi zformuloval další slova, velitel se
            ušklíbl:
            <.p><q>Nebuď zvědavej, budeš brzo starej.</q>
            <.reveal default-topic> '
    ]
;

+ DefaultAnyTopic, ShuffledEventList
    [
        '<.p>Pokusil ses oslovit velitele, ale ten zívá a vůbec tě nevnímá.
            <.reveal default-topic> ',
        '<.p><q>Veliteli, mohl bych…?</q> začal jsi mluvit, ale velitel tě
            otráveně přerušil:
            <.p><q>Co jsi říkal?</q><.reveal default-topic> '
    ]
;

/* ------------------------------------------------------------------------ */
/*
 *   UVÍTÁNÍ NA ZÁKLADNĚ A VYSVĚTLENÍ PROBLÉMŮ
 *
 *   Tento stav a konverzační uzel 'welcome' níže je aktivován po společném
 *   výstupu z lodi (captainGuideToDomeInTravel).
 */
+ commanderWelcomes: ActorState
    stateDesc = "Je plný nadšení z vašeho příletu. "

    /*
     *   Úmyslně ponecháme specialDesc prázdný, protože ho popíšeme skupinově
     *   v rámci kapitánova stavu. Tím se dosáhne skupinového efektu místo
     *   samostatného odstavce.
     */
    specialDesc = ""
;

/* ------------------------------------------------------------------------ */
/*
 *   Konverzační uzel představuje bod v konverzaci, kdy se nás velitel základny
 *   vítá a pokládá nám otázku. Uzel tedy řeší naše možné odpovědi.
 */
+ ConvNode 'welcome'

    /*
     *   Hráče v uzlu podržíme (isSticky) tak dlouho, dokud neodpoví. Odpovědět
     *   může jen ano či ne (YesTopic a NoTopic níže). Protože je hra určena
     *   začátečníkům, necháme vypsat doporučené odpovědi tagem <.topics> v
     *   textu a protože by mohlo být více doporučených témat, omezíme nabízené
     *   možnosti konverzace jen na ty konverzačního uzlu (limitSuggestions).
     */
    isSticky = true
    limitSuggestions = true

    /* V tomto uzlu je to velitel, kdo oslovuje hráče a ptá se ho. */
    npcGreetingMsg = "<.p><q>Vítejte, chlapi. Vyhlížíme vás na radaru už druhý
        den,</q> povídá velitel a potřásá si rukou s kapitánem, <q>ahoj, Henry,
        koukám, že s novým parťákem.</q>

        <.p><q>Ahoj, základno, to je tu taková nuda, že jste se nás nemohli
        dočkat?</q> žertuje kapitán, <q>zdravím, Ivane, tohle je Tobiáš, mladá
        krev na našem vraku.</q>

        <.p><q>Plný humoru jako vždycky,</q> povídá velitel a obrací se k tobě,
        <q>cesta proběhla v pohodě?</q><.topics> "

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku. Snažíme se ho udržet v konverzaci,
     *   kterou začal velitel.
     */
    npcContinueList: ShuffledEventList
    {
        [
            '<.p><q>Tobiáši, vnímáš nás vůbec? Ten kluk snad neslyší,</q>
                stěžuje si kapitán veliteli, <q>haló, Tobiáši!</q> ',
            '<.p><q>Koukni na toho kluka,</q> upozorňuje kapitán velitele, <q>on
                snad neslyší, že se ho na něco ptáš. Haló, Tobiáši!</q> ',
            '<.p><q>Koukni na toho kluka, on má snad klapky na uších a hraje si
                na Otíka!</q> povzdechl si kapitán, <q>a to po něm nechceš nic
                víc, než jestli proběhla cesta pohodě.</q> '
        ]

        /*
         *   Hlášky zobrazujeme s 50% pravděpodobností a poprvé je nezamícháme,
         *   protože jsou odstupňované.
         */
        eventPercent = 50
        shuffleFirst = nil
    }

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
    {
        switch(reason)
        {
        case endConvBye:
            /* Proc tohle nenastava? Zvoli se vzdy DefaultAnyTopic. */
            "<.p>Hodil jsi neurčitě do prostoru nashledanou, ale zastavilo tě
                ostřejší kapitánovo upozornění: <q>Tobiáši, ještě jsme
                nedomluvili, velitel Petronov se ptal, zda cesta proběhla v
                pořádku.</q> ";
            return blockEndConv;
        case endConvTravel:
            "<.p>Rád bys odešel, ale slušné vychování ti brání odejít vprostřed
                hovoru. Velitel se přece ptal, jestli cesta byla v pohodě. ";
            return blockEndConv;
        default:
            return nil;
        }
    }

    /*
     *   Možnost odpovědět ano či ne je spíše pro dojem interaktivity, linie
     *   konverzace se zase spojí.
     */
    commonFollowup = "<.p><q>To mě u té vaší kraksny nepřekvapuje,</q> usmívá se
        velitel, <q>pamatuješ, jak vám tenkrát shořela 3kx27, byli jste na
        dohled a vlekli jste se sem 3 týdny? Co to bylo tentokrát?</q>

        <.p><q>Domníváme se, že to asi bylo…,</q> kapitán si promnul nos, <q>že
        došlo ke zkratu v radarovém modulu.</q>

        <.p><q>Porouchalo nám to navigaci,</q> rychle přidáváš podrobnosti,
        <q>málem jsme přistáli na jiném šutru.</q>

        <.p><q>Ale jste tady, to je hlavní, živí a zdraví,</q> konstatuje s
        úlevou velitel. <q>Ale bez opravy asi neodletíte. Chcete se do toho
        pustit hned?</q>

        <.convnode is-critical><.topics><.reveal technician-incoming> "
;

++ YesTopic, SuggestedYesTopic
    "<.p><q>No jo,</q> nadechl ses k odpovědi, ale slova se ujal kapitán:
        <.p><q>Prospal jsem konec cesty, ale tady mladej si dokázal poradit,</q>
        konstatoval spokojeně a s trochou hrdostí v hlase.

        <.p><q>Ten podivný zkrat před přistáním mě docela vyvedl z míry,</q>
        doplňuješ kapitána, <q>ale nakonec jsem to zvládl!</q>
        <<location.commonFollowup>> "
;

++ NoTopic, SuggestedNoTopic
    "<.p><q>No,</q> nadechl ses k odpovědi, ale do řeči ti skočil kapitán:
    
        <.p><q>Táhnul jsem hodiny služby a jen jsem zalehl, tak tady mladej nás
        málem zabil,</q> pobaveně sděluje.

        <.p><q>To já neovlivnil,</q> ohrazuješ se, <q>těsně před příletem k vám
        došlo k nějakému zkratu a já si s tím sám dokázal poradit!</q>
        <<location.commonFollowup>> "
;

++ HelloTopic
    "<.p>Už jste se přivítali a teď se tě velitel ptá, zda cesta proběhla v
        pořádku. "
;

++ ByeTopic
    "<.p>Hodil jsi neurčitě do prostoru nashledanou, ale zastavilo tě ostřejší
        kapitánovo upozornění: <q>Tobiáši, ještě jsme nedomluvili, velitel
        Petronov se ptal, zda cesta proběhla v pořádku.</q> "
;

++ DefaultAnyTopic
    "<.p>Zapomínáš na slušné vychování, velitel se tě ptal, jestli cesta
        proběhla v pořádku. Měl bys mu odpovědět po pravdě, jestli ano či ne. "
;

/* ------------------------------------------------------------------------ */

+ ConvNode 'is-critical'
    isSticky = true
    limitSuggestions = true

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku.
     */
    npcContinueList: ShuffledEventList
    {
        [
            '<.p><q>Tobiáši, vnímáš nás vůbec? Ten kluk snad neslyší,</q>
                stěžuje si kapitán veliteli, <q>haló, Tobiáši!</q> ',
            '<.p><q>Koukni na toho kluka,</q> upozorňuje kapitán velitele, <q>on
                snad neslyší, že se ho na něco ptáš. Haló, Tobiáši!</q> ',
            '<.p><q>Koukni na toho kluka, on má snad klapky na uších a hraje si
                na Otíka!</q> povzdechl si kapitán, <q>a to po něm nechceš nic
                víc, než jestli proběhla cesta pohodě.</q> '
        ]
        eventPercent = 50
        shuffleFirst = nil
    }

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
    {
        switch(reason)
        {
        case endConvBye:
            "<.p>Hodil jsi neurčitě do prostoru nashledanou, ale zastavilo tě
                ostřejší kapitánovo upozornění: <q>Tobiáši, ještě jsme
                nedomluvili, velitel Petronov se ptal, zda se pustíš do opravy
                lodi.</q> ";
            return blockEndConv;
        case endConvTravel:
            "<.p>Rád bys odešel, ale slušné vychování ti brání odejít vprostřed
                hovoru. Velitel se přece ptal, zda se pustíš do opravy lodi. ";
            return blockEndConv;
        default:
            return nil;
        }
    }

    /*
     *   Teď teprve opustíme doposud přilepené konverzační uzly, technika
     *   necháme doprovodit hráče do dílny a kapitán s velitelem zde zatím
     *   zůstanou ve společném hovoru, než hráč odejde s Borisem.
     *
     *   Protože se mluvilo o vadném modulu, tak i když ho zatím hráč nespatřil,
     *   musíme ho označit za známý, aby se na něj mohl hráč ptát. Normálně se
     *   totiž nemůže ptát na něco, co ještě nepotkal.
     */
    commonFollowup = "<.p><q>Tohle je náš technik, Boris,</q> otáčí se velitel k
        technikovi, <q>vezme mladýho do dílny a my si zatím vyřídíme
        papírování.</q>

        <.p><q>Ty jsi Tobiáš?</q> podává ti Boris ruku, <q>tak pojď, koukneme, s
        čím ti naše vybavení může pomoct</q> a směruje tě k žebříku.

        <.convnode nil>
        <<technician.setCurState(technicianGuideToCorridorWest)>>
        <<captain.setCurState(captainChatting)>>
        <<commander.setCurState(commanderChatting)>>
        <<gSetKnown(firstModule)>> "
;

++ YesTopic, SuggestedYesTopic
    "<.p><q>Raději ano,</q> okamžitě reaguješ, <q>potřebuju vytáhnout zvenku
        moduly navigace a podívat se, co vlastně všechno shořelo.</q>

        <.p><q>Správný přístup, mladej,</q> s úsměvem tě chválí kapitán, <q>dej
        se do toho!</q><<location.commonFollowup>> "
;

++ NoTopic, SuggestedNoTopic
    "<.p><q>Opravit to musíme. Bude se muset z venku k lodi,</q> oznamuješ bez
        nadšení, <q>ale myslím, že na to nemusíme spěchat. Stačí po obědě.</q>

        <.p><q>Co můžeš udělati hned… Hele, mladej, raději, ať víme, na čem
        jsme,</q> popohání tě kapitán, <q>dej se do práce!</q>
        <<location.commonFollowup>> "
;

++ HelloTopic
    "Už jste se přivítali a teď se tě velitel ptá, zda se pustíte do opravy
        lodi. ";

++ ByeTopic
    "<.p>Hodil jsi neurčitě do prostoru nashledanou, ale zastavilo tě ostřejší
        kapitánovo upozornění: <q>Tobiáši, ještě jsme nedomluvili, velitel
        Petronov se ptal, zda se pustíš do opravy lodi.</q> ";

/*
 *   Tyto hlášky zobrazujeme, když se hráč pokouší komunikovat o něčem jiném,
 *   než na co byl dotázán.
 */
++ DefaultAnyTopic
    "<.p>Zapomínáš na slušné vychování, velitel se tě ptal, jestli se pustíš do
        opravy lodi. Měl bys mu odpovědět, jestli ano či ne. "
;

/* ------------------------------------------------------------------------ */
/*
 *   PO SKONČENÍ UVÍTÁNÍ
 */
+ commanderChatting: ActorState
    stateDesc = "Polohlasem se baví s kapitánem o novinkách a soukromých
        záležitostech. "

    /*
     *   Úmyslně ponecháme specialDesc prázdný, protože ho popíšeme skupinově
     *   v rámci kapitánova stavu. Tím se dosáhne skupinového efektu místo
     *   samostatného odstavce.
     */
    specialDesc = ""
;

++ HelloTopic "Sice už hovoří o něčem jiném, ale klidně na ně můžeš mluvit. ";
++ ByeTopic "<.p><q>Pánové, odcházím zdechnout do houští,</q> obrátil ses k
    veliteli a kapitánovi, <q>teda jdu se podívat na tu vaši dílnu.</q> ";

/*
 *   Zde máme specifický DefaultTopic kvůli navození atmosféry, že se velitel
 *   baví s kapitánem. To ale znamená, že zaujme roli každého tématu, které zde
 *   nenalezne a tím by překryla i všechna témata, které nejsou zde v
 *   commanderChatting, což nechceme. Proto pomocí deferToEntry řekneme, že
 *   jiná, než DefaultTopic témata se mají hledat v nadřazené databázi témat,
 *   tj. ne zde ve stavu postavy, ale přímo u postavy.
 */
++ DefaultAnyTopic
    "<.p>Vypadá to, že jsou příliš zabráni do společného rozhovoru, že tě ani
        nevnímají. "

    deferToEntry(entry) { return !entry.ofKind(DefaultTopic); }
;

/* ------------------------------------------------------------------------ */
/*
 *   VELITEL JDE S KAPITÁNEM DO KAJUTY
 *
 *   Během cesty se velitel nenechá rušit, to zařídí HermitActorState. Cestu
 *   řídí TravelAgendaItem a aktivujese ihned, jak zapneme stav
 *   commanderGoToCommandersRoom.
 */
+ commanderGoToCommandersRoom: HermitActorState
    specialDesc = ""
    noResponse = "Jsou příliš zabraní do společného rozhovoru. "

    /* Potlačíme standardní výpis, cesta je popsána v rámci kapitána. */
    sayArrivingDownStairs(conn) { }
    sayDepartingLocally(dest, conn) { }
    sayArrivingLocally(dest, conn) { }
    sayDepartingThroughPassage(conn) { }
    sayTravelingRemotely(dest, conn) { }
    sayDepartingDir(dir, conn) { }
;

+ commanderGoToCommandersRoomAgenda: TravelAgendaItem
    isReady = (commander.curState == commanderGoToCommandersRoom)
    initiallyActive = true
    travelPath = [dome, corridorWest, corridorEast, commandersRoom]

    /* Jakmile dorazí do kajuty, přepneme stav postavy. */
    destReached()
    {
        commander.setCurState(commanderPaperWork);
        gReveal('commander-in-his-room');
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN S VELITELEM VYŘIZUJÍ PAPÍRY V KAJUTĚ
 *
 *   AgendaItem je něco, co má postava v plánu za určitých podmínek říci či
 *   udělat, když se naskytne příležitost. Zde potřebujeme, aby jakmile za nimi
 *   hráč poprvé přijde do kajuty, aby před ním narychlo ukryli papíry do
 *   trezoru a tím v hráči vzbudili podezření. Podmínka v isReady zjišťuje, zda
 *   je ten správný okamžik agendu vyvolat.
 */
+ commanderHidingAgenda: AgendaItem
    initiallyActive = true
    isReady = commander.curState == commanderPaperWork && commander.canSee(me)

    invokeItem()
    {
        "<.p>Pánové sebou trhli a kapitán rychle shrábl listiny rozložené po
            stole. Když jsi udělal krok k nim, velitel vstal, listiny vložil do
            trezoru a prudce bouchl jeho dvířky: <q>Co se děje, mladíku?</q> ";

        /* Je to jen jednorázová záležitost, označíme agendu za vyřízenou. */
        isDone = true;

        gReveal('hint-hiding');
    }
;

/*
 *   Když kapitán s velitelem dorazí do kajuty, tak se chovají v podstatě jako
 *   jedna postava, čili hráč mluví s oběma najednou a jejich stavy konverzace
 *   se synchronizují. Společná konverzační témata jsou v samostatném souboru
 *   actorCommonTopics.t, Hello a Goodbye mají každý své.
 */
+ commanderPaperWorkTalking: ProxyTopicDatabase, InConversationState
    stateDesc = "Odložil papíry, aby se s tebou mohl bavit. "

    /*
     *   Úmyslně ponecháme specialDesc prázdný, protože ho popíšeme skupinově
     *   v rámci kapitánova stavu. Tím se dosáhne skupinového efektu místo
     *   samostatného odstavce.
     */
    specialDesc = ""

    /* Dokud hráč neodejde či se nerozloučí, věnují mu pozornost neustále. */
    attentionSpan = nil

    /* Společná databáze odpovědí pro obě postavy. */
    proxyTarget = commonTopics
;

/*
 *   Dvojice InConversationState a ConversationReadyState spolu s HelloTopic a
 *   ByeTopic zajišťuje navazování a odchod z konverzace. Vloží se do postavy v
 *   tomto uspořádání.
 */
++ commanderPaperWork: ConversationReadyState
    stateDesc = "Probírá se papíry. "

    /*
     *   Úmyslně ponecháme specialDesc prázdný, protože ho popíšeme skupinově
     *   v rámci kapitánova stavu. Tím se dosáhne skupinového efektu místo
     *   samostatného odstavce.
     */
    specialDesc = ""
;

/*
 *   Jestliže mám v EventListu dva seznamy, první se nejprve jednou probere
 *   v pořadí definice a pak se teprve dělá běžný ShuffledEventList se druhým
 *   seznamem.
 */
+++ HelloTopic, ShuffledEventList
    [
        '<.p><q>Ehm,</q> odkašlal sis, <q>omlouvám se, že vás ruším.</q>

            <.p><q>No, jsi tady, tak povídej, co potřebuješ,</q> vyzval tě
                velitel. '
    ]
    [

        '<.p>Tvůj pozdrav je očividně nevhod -- oba pánové velmi zaujatě
            probírají papíry na stole: <q>Potřebuješ něco, Tobiáši?</q> zeptal
            se velitel, aniž by zvedl hlavu. ',

        '<.p><q>Zdravím,</q> zahalekal jsi docela oprskle.
            <.p>Oba pánové zvedli hlavu a kapitán kývnul tvým směrem: <q>Co se
            děje, mladej?</q> '
    ]
;

+++ ByeTopic, ShuffledEventList
    [
        '<.p><q>Tak já zas jdu,</q> přidržel ses dveří, ale už ti nikdo nevěnuje
            pozornost, pánové mají zajímavější téma, než jsou tvoje problémy. ',

        '<.p><q>Zatím,</q> rozloučil ses a oba pánové ti kývli v odpověď.
            Očividně vítají, že bys mohl konečně odejít. ',

        '<.p><q>Tak já jdu,</q> oznámil jsi.

            <.p><q>A koukej něco dělat,</q> zareagoval na tvé rozloučení kapitán
            a už tě nevnímá, obrátil se k veliteli. '
    ]
;

++ AskTellTopic @suspiciousContainer
    "<.p>Už se chystáš vyslovit dotaz, ale něco tě zarazilo. Dal bys ruku do
        ohně, že jeden z těch dvou u stěhování bedny byl velitel. Raději se
        koukni jinde. "

    /*
     *   Protože tohle vlastně není konverzace, hráč si rozmyslí a nepoloží
     *   otázku, tak musíme označit, aby neproběhlo automatické pozdravení.
     */
    isConversational = nil
;

++ AskTellTopic @tChriss
    "<.p>Obrátil ses k veliteli: <q>Pane, jistě jste se setkal s předchozím
        parťákem kapitána Scotta, s Chrissem Andersonem. Řekněte mi prosím něco
        o něm.</q>

        <.p><q>A co ti mám říci,</q> povzdechl velitel, <q>sice sem létal s
        Henrym dlouho, pozdravili jsme se u jídla a to bylo všechno.</q> "
;

/* ------------------------------------------------------------------------ */
/*
 *   PO NÁVRATU VELITEL DOPROVÁZÍ HRÁČE DO JÍDELNY
 *
 *   Velitel nás vede do jídelny, tedy spíše nám dává přednost ve dveřích. Na
 *   nás je, abychom přesun mezi lokacemi zahájili. Každý přechod mezi lokacemi
 *   se skládá z dvojice stavů, jeden pro čekání na nás a druhý pro samotný
 *   přechod mezi místnostmi.
 */
+ commanderGuideToCanteen: GuidedTourState
    stateDesc = "Čeká u vchodu do jídelny, až se k němu připojíš. "
    specialDesc { inherited; stateDesc; }

    escortDest = canteenPassageOutside
    escortStateClass = CommanderGuideToCanteenInTravel

    stateAfterEscort = commanderInCanteen
;

class CommanderGuideToCanteenInTravel: GuidedInTravelState
    sayDeparting(conn) { "Přidržel ses madel a lehce vklouzl do jídelny. "; }
    specialDesc()
    {
        "<.p>Rozhlédl ses po jídelně. Ano, normálka, stravování jako ve školním
            výcviku. Došel jsi k okénku a vzal potravinový balíček s obědem.
            Hm, asi nějaké patlámo, těstoviny, klasické žrádlovaky: <q>Koukám,
            že hydroponie funguje,</q> usmál ses na velitele, <q>jahody jsem v
            poslední době viděl jen na obrázku.</q>

            <.p><q>Ať ti chutná,</q> popřál ti dobrou chuť velitel a přidali se
            k němu i ostatní, jen co ses usadil u stolu. ";

        /* Postavy se nesmí přemisťovat pouhým moveInto, jako běžné předměty. */
        me.moveIntoForTravel(bench);
        me.makePosture(sitting);

        commander.moveIntoForTravel(bench);
        commander.makePosture(sitting);

        /* Tác s jídlem je PresentLater objekt, necháme ho objevit. */
        platter.makePresent();

        /* Načasujeme událost na jeden tah dopředu. */
        fixedFuseId = new Fuse(self, &fixedFuse, 1);
    }

    /*
     *   Fuse je událost, která se jednorázově stane po určitém počtu tahů. Zde
     *   definujeme funkci, která se vyvolá, samotné načasování je výše.
     *
     *   Úkolem je zahájit rozhovor, tedy aby kapitán hráče oslovil a předal mu
     *   opravené součástky. Se zpožděním pomocí Fuse se trápíme hlavně proto,
     *   aby kapitánovo oslovení se nestalo hned ve stejném tahu, kdy hráč vejde
     *   do jídelny a sedne si (zde už je dlouhý text). Tím, hráči ale dáme
     *   příležitost po usednutí zadat nějaký příkaz, tak se může hráč začít na
     *   něco ptát. Ani v té situaci ještě nechceme, aby kapitán hned hráče
     *   oslovil svou větou. Proto odkládáme novým a novým časováním té samé
     *   Fuse začátek rozhovoru tak dlouho, doku se hráč nezačne věnovat něčemu
     *   jinému.
     */
    fixedFuseId = nil
    fixedFuse()
    {
        if(captain.conversedThisTurn())
            fixedFuseId = new Fuse(self, &fixedFuse, 1);
        else
            captain.initiateConversation(captainInCanteenTalking,
                'fixed-components');
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN S VELITELEM V JÍDELNĚ
 */
+ commanderInCanteen: HermitActorState
    noResponse = "Zrovna má plnou pusu jídla. "
    stateDesc = "Něco s chutí jí. "
    /* Nechame umyslne prazdne. */
    specialDesc = ""
;

/* ------------------------------------------------------------------------ */
/*
 *   ROZHOVOR PŘES VYSÍLAČKU PO ODJEZDU ZE ZÁKLADNY
 *
 *   Zdaleka nejsložitější a pro hru nejdůležitější rozhovor se odehrává přes
 *   vysílačku, když hráč jede do zakázané oblasti. Skládá se z mnoha uzlů
 *   konverzace a v každém se důsledně hlídá, co hráč může udělat, aby byl držen
 *   v linii rozhovoru až do závěrečného zvratu.
 */
+ commanderOnRadio: ActorState;

/* ------------------------------------------------------------------------ */
/*
 *   Nejprve volá velitel hráče a požaduje po něm, aby se ozval. Tak či onak se
 *   hráč nakonec musí ohlásit.
 */
+ ConvNode 'do-you-read-me'
    npcGreetingMsg()
    {
        "<.p><q>Kchh prask chrchle chrapl…,</q> ozvalo se z vysílačky.
            <q>Základna volá průzkumné vozidlo Červ 2. Ohlaš se,</q> zní v rádiu
            velitel. A po chvíli znovu: <q>Tady velitel, volám Červa, ohlaš se.
            Mám signál, že jsi opustil základnu.</q><.reveal hint-avoid> ";

        clock.eventReached();
    }
    clock: ClockEvent { eventTime = [2, 14, 26] }

    /* V tomto konverzačním uzlu nenabízíme jiná témata. */
    limitSuggestions = true
    isSticky = true

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
        { return nil; }

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku.
     */
    npcContinueList: ShuffledEventList
    {
        [
            '<q>Haló, Tobiáši, ozvi se,</q> opakuje velitelův hlas monotónně ve
                vysílačce, jakoby to byl stařičký gramofon se zaseknutou
                jehlou. ',
            '<q>Základna volá Červa! Červe, ozvi se,</q> a po chvíli znova:
                <q>základna volá Červa, Červe, ozvi se!</q> ',
            '<q>Tobiáši, potvrď příjem, Tobiáši, ozvi se,</q> slyšíš opakovat
                velitelův hlas ve vysílačce. ',
            'Ta vysílačka ti jde na nervy: To je pořád <q>Ozvi se! Haló! Slyšíš
                mě, Tobiáši?</q> '
        ]

        /* Zobrazujeme je s cca 2/3 pravděpodobností. */
        eventPercent = 67
    }

    /*
     *   Hráč se může ohlásit různými způsoby. Má samozřejmě speciální téma
     *   "ohlaš se", ale může i jen pozdravit či říci ano, protože se ptají, zda
     *   je slyší. Všechny možnosti směřují sem, kde se odehraje společná část
     *   rozhovoru a posunutí se na další konverzační uzel.
     */
    commonFollowup = "<.p>Poznáváš velitelův značně podrážděný hlas: <q>Nemůžeš
        jen tak opustit základnu. Koukej se hned vrátit! Co tě to vlastně
        napadlo sebrat vozítko a odjet pryč?</q>

        <.p><q>Kruci a kruci,</q> pomyslel sis, takhle brzo jsi vyhmátnutí
        nečekal. Bude to chtít získat trochu času.<.convnode why-leaved> "
;

++ HelloTopic
    "<.p>Po chvíli váhání odpovídáš: <q>Tady Tobiáš.</q>

        <<location.commonFollowup>> "
;

/*
 *   SpecialTopic je jednorázová možnost jak v rozhovoru pokračovat, která
 *   se zcela vymyká z běžného schématu zeptej se na/řekni o. Proto musíme
 *   poskytnout vyčerpávající slovník, jak hráč může zareagovat. Jakákoliv
 *   odpověď hráče, která se skládá pouze ze slov vyjmenovaných níže, se
 *   bere jako schodující se.
 */
++ SpecialTopic 'se ohlásit'
    ['ohlaš', 'ohlásit', 'ozvi', 'ozvat', 'odpověz', 'odpovědět', 'se', 'mu',
        'jim', 'veliteli', 'kapitánovi', 'základně']

    "<.p>Po chvíli váhání odpovídáš: <q>Tady Tobiáš.</q>

        <<location.commonFollowup>> "
;

++ YesTopic
    "<.p>Po chvíli váhání odpovídáš: <q>Tady Tobiáš.</q>

        <<location.commonFollowup>> "
;

/*
 *   Nabízíme i nějaké další možnosti, než se ohlásit, aby hráč měl pocit, že se
 *   rozhoduje, ale ve skutečnosti vedou stejným směrem.
 */
++ SpecialTopic 'je ignorovat'
    ['ho', 'je', 'velitele', 'kapitána', 'ignoruj', 'ignorovat']

    "<.p>Pane kopilote, přestaňte nás ignorovat a okamžitě se ohlaste!

        <.p>Nemá smysl si dál hrát na hluchého a odpovídáš: <q>Tady Tobiáš.</q>
        <<location.commonFollowup>> "
;

++ NoTopic
    "<.p>Pane kopilote, přestaňte nás ignorovat a okamžitě se ohlaste!

        <.p>Nemá smysl si dál hrát na hluchého a odpovídáš: <q>Tady Tobiáš.</q>
        <<location.commonFollowup>> "
;

++ DefaultAnyTopic
    "<.p>Něco nesrozumitelného jsi zahudral do vysílačky.
        <<location.commonFollowup>> "
;

/* ------------------------------------------------------------------------ */
/*
 *   V tomto konverzačním uzlu se velitel ptá hráče, proč vlastně odjel. Opět
 *   nabízíme různé možnosti odpovědí, ale ty zase směřují stejným směrem.
 */
+ ConvNode 'why-leaved'

    /* V tomto konverzačním uzlu nenabízíme jiná témata. */
    limitSuggestions = true
    isSticky = true

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
        { return nil; }

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku.
     */
    npcContinueList: ShuffledEventList
    {
        [
            '<q>Tobiáši, na něco jsem se tě ptal!</q> opakuje velitel z
                vysílačky, <q>odpověz mi na otázku!</q> ',
            '<q>Haló, slyšíme se? Základna volá Červa, ptal jsem se,</q>
                připomíná velitelův hlas, <q>proč jsi odjel pryč?</q> ',
            '<q>Haló, haló,</q> opakuje pořád hlas z vysílačky, <q>základna pro
                Červa. Tobiáši, odpověz, co tě to napadlo odjet?</q> ',
            '<q>Tobiáši, to neumíš odpovědět?</q> nechtěl bys teď stát přímo u
                velitele, <q>ptal jsem se snad dost jasně. Co tě to popadlo
                sebrat vozítko?!</q> '
        ]
        eventPercent = 67
    }
;

++ HelloTopic "Už spolu komunikujete. ";

++ ByeTopic, StopEventList
    [
        'Řekl jsi do vysílačky: <q>Končím!</q> ale se zlou ses potázal,
            velitelův hlas tě poučil o zásadách slušného chování tak důrazně, že
            lituješ, žes vůbec otvíral ústa. ',
        'Pokusil ses hovor ukončit, ale jednak se ti vůbec nepodařilo skočit
            veliteli do řeči a jednak vysílačka nejde vypnout. '
    ]
;

++ SpecialTopic 'se vymluvit, že se jen rozhlížíš po okolí základny'
    ['vymluv', 'vymluvit', 'řekni', 'říci', 'říct', 'že', 'se', 'jen',
        'rozhlížíš', 'rozhlížím', 'rozhlíží', 'po', 'základně', 'okolí',
        'základny', 'o', 'tom', 'výmluvě']

    "<.p><q>Už mám hotovou opravu lodi, tak jsem si říkal, že se rozhlídnu po
        okolí základny,</q> zcela přesvědčivým hlasem zahlásíš do vysílačky.

        <.p><q>To jsi, chlapče, trochu zabloudil,</q> ironie v hlasu velitele
        přímo bodá, <q>signál z tvého vozítka je na takovou vycházku pěkně
        daleko. Přiznej barvu! Někomu jsi tady na základně sebral kartu, jinak
        by ses do vozítka nedostal. Tak co máš v plánu, kam chceš jet?</q>
        <.convnode destination><.reveal hint-destination> "
;

++ SpecialTopic 'říci, že se jedeš podívat na místo těžby'
    ['řekni', 'říci', 'říct', 'že', 'jedeš', 'jedu', 'jede', 'jdeš', 'jdu',
        'jde', 'podívat', 'se', 'na', 'těžbu', 'místo', 'místě', 'těžby', 'o',
        'tom', 'těžbě', 'podívání', 'těžbě', 'těžba']

    "<.p><q>Už mám hotovou opravu lodi, tak jsem chtěl ještě využít čas, než
        odletíme, a podívat se přímo do dolu na těžbu,</q> oznámil jsi do
        vysílačky doufaje, že tvůj hlas se nechvěje.

        <.p><q>To vyprávěj své babičce,</q> úplně jsi viděl velitelův úšklebek,
        <q>jedeš zcela opačným směrem. Buď jsi propadl z navigace nebo nám
        vyprávíš báchorky. Nedělej z nás blbce, někomu tady na základně jsi
        šlohnul kartu, jinak by ses do vozítka nedostal. Kam vlastně chceš
        jet?</q><.convnode destination><.reveal hint-destination> "
;

++ DefaultAnyTopic, ShuffledEventList
    [
        '<.p>Sotva jsi začal, velitel se ozval:
            <.p><q>Hergot, Tobiáši, tady kladu otázky já,</q> důrazně říká z
            vysílačky, <q>tak znova, proč jsi odjel ze základny?</q> ',
        '<.p><q>Já bych chtěl…</q> začal jsi mluvit, ale velitel ti hned skočil
            do slova:
            <.p><q>To mě teď vůbec nezajímá, že ty něco chceš,</q> rozčiluje se,
            <q>řekni mi konečně, proč jsi odjel ze základny!</q> ',
        '<.p><q>Tak by mě zajímalo,</q> začal jsi, ale na druhé straně se ozval
            velitelův hlas:
            <.p><q>Neptej se na hlouposti, mladej,</q> odsekává z vysílačky,
            <q>a raději mi řekni, proč jsi takhle potichounku zmizel ze
            základny!</q> ',
        '<.p>Začal jsi mluvit, ale než jsi stihl dopovědět, už tě velitel
            přerušil:
            <.p><q>Nech těch blbech řečí a odpověz,</q> syknul z vysílačky. '
    ]
;

/* ------------------------------------------------------------------------ */
/*
 *   Když hráč prozradí, kam jede, zeptá se velite proč tam jede. Nervozita
 *   narůstá a velitel začíná tušit, že hráč strká nos tam, kam nemá a že hrozí
 *   prozrazení tajemství. Zatím ale přesně neví, kolik toho hráč ví a proto se
 *   ptá a snaží se hráče od cesty odradit.
 */
+ ConvNode 'destination'

    /* V tomto konverzačním uzlu nenabízíme jiná témata. */
    limitSuggestions = true
    isSticky = true

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
        { return nil; }

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku.
     */
    npcContinueList: ShuffledEventList
    {
        [
            '<q>Tobiáši, na něco jsem se tě ptal!</q> opakuje velitel z
                vysílačky, <q>odpověz mi na otázku!</q> ',
            '<q>Haló, slyšíme se? Základna volá Červa, ptal jsem se,</q>
                připomíná velitelův hlas, <q>kam vlastně jedeš?</q> ',
            '<q>Haló, haló,</q> opakuje pořád hlas z vysílačky, <q>základna pro
                Červa. Tobiáši, odpověz, kam vlastně chceš jet?</q> ',
            '<q>Tobiáši, to neumíš odpovědět?</q> nechtěl bys teď stát přímo u
                velitele, <q>ptal jsem se snad dost jasně. Kam chceš ksakru
                jet?!</q> '
        ]
        eventPercent = 67
    }
;

++ HelloTopic "Už spolu komunikujete. ";

++ ByeTopic, StopEventList
    [
        'Řekl jsi do vysílačky: <q>Končím!</q> ale se zlou ses potázal,
            velitelův hlas tě poučil o zásadách slušného chování tak důrazně, že
            lituješ, žes vůbec otvíral ústa. ',
        'Pokusil ses hovor ukončit, ale jednak se ti vůbec nepodařilo skočit
            veliteli do řeči a jednak vysílačka nejde vypnout. '
    ]
;

/*
 *   Zde je jeden z možných konců hry, pokud se hráč nechá odradit a vrátí se,
 *   aniž by zakázanou oblast navštívil.
 */
++ SpecialTopic 'se vrátit'
    ['řekni', 'říci', 'říct', 'že', 'se', 'vracíš', 'vrátíš', 'vrať', 'vrátit',
        'vracím']

    topicResponse()
    {
        "<.p><q>Už nikam,</q> rezignovaně ses ozval do vysílačky, <q>vracím
            se.</q> Naklonil ses k ovládání a stiskl tlačítko <q>domů</q>.

            <.p>Po chvíli ticha z vysílačky odkáplo jen stručné:
            <q>Dobře.</q>";

//        inputManager.pauseForMore(true);

        "<.p>Dojel jsi na základnu, vystoupil z vozítka a ani tě nepřekvapilo, že
            na tebe čekají oba velitelé. Ten ze základny jen mlčky natáhl ruku,
            do které jsi vložil technikovu kartu, a kapitán Scott tě nepříliš
            jemně postrčil směrem k vaší lodi. Ani jsi nevěděl, že kajuta jde
            zamknout zvenčí. Cesta zpátky se vlekla, i když jsi po nějaké době
            nastoupil do služby a střídal se s kapitánem, vládlo v lodi ledové
            mlčení. Pořád v tobě přetrvával pocit, žes někde něco prošvihnul.

            <.p>Po návratu na Zem ses pokoušel kontaktovat pár lidí se svými
            podezřeními, ale všude ti bylo řečeno, že to jsou jen tvé doměnky bez jakéhokoliv důkazu.
            A další cestovní příkaz k cestě do vesmíru nepřichází. ";

/* Když hrály děti soutěžně, byl na webu žebříček v reálném čase. */
#ifdef TADS_INCLUDE_NET
//        local data =
//        [
//            'sid' -> webSession.storageSID,
//            'event' -> 'surrender2',
//            'points' -> 0,
//            'desc' -> '',
//            'total' -> libScore.totalScore
//        ];

//        sendNetRequest(0, 'http://technoplaneta.cz/t3submit', 'POST', 0, nil,
//            data);
#endif

        /* Jeden z neúspěšných konců. */
        finishGameMsg('VZDAL SES', [ finishOptionUndo, finishOptionCredits,
            finishOptionFullScore ]);
    }
;

/*
 *   Konečně se situace vyhrocuje. Hráč už ví, že nemá cenu zapírat a je tlačen
 *   najít nějaké východisko. Snažíme se ho motivovat k tomu, aby převzal
 *   iniciativu a aby se on začal ptát, aby on řídit rozhovor.
 */
++ SpecialTopic 'říci pravdu'
    ['řekni', 'říci', 'říct', 'pravdu']

    "<.p>Zhluboka ses nadechl, zatnul pěsti a oznámil: <q>Chci se podívat na
        souřadnice SQ71.</q>

        <.p><q>Aha,</q> podivil se hlas tam na druhé straně, <q>co si myslíš, že
        tam najdeš? Když víš souřadnice, víš také, že to je zakázaná oblast.</q>

        <.p>Honem hledáš vhodná slova, jak se bránit, když velitel pokračuje:
        <q>Porušuješ v tuto chvíli tolik paragrafů, že by to stačilo na dva
        životy bez letů do vesmíru. Jseš si vědom, že…</q>

        <.p>Přestal jsi vnímat, ta útočná a výhružná slova z vysílačky kolem
        tebe plynou zcela beze smyslu, ale to ostatní najednou začne dávat
        smysl: K čemu tolik křiku, pokud by tam nebylo něco, co bych neměl
        vidět? Neměli by spíše oni něco vysvětlovat? Proč se mám omlouvat já,
        když jsem nic neprovedl. Teda, pokud nepočítám tohle <q>půjčené</q>
        vozítko. Možná je čas na pár otázek.<.convnode questions><.topics>
        <.reveal hint-questions> "
;

++ DefaultAnyTopic, ShuffledEventList
    [
        '<.p>Sotva jsi začal, velitel se ozval:
            <.p><q>Hergot, Tobiáši, tady kladu otázky já,</q> důrazně říká z
            vysílačky, <q>tak znova, proč jsi odjel ze základny?</q> ',
        '<.p><q>Já bych chtěl…</q> začal jsi mluvit, ale velitel ti hned skočil
            do slova:
            <.p><q>To mě teď vůbec nezajímá, že ty něco chceš,</q> rozčiluje se,
            <q>řekni mi konečně, proč jsi odjel ze základny!</q> ',
        '<.p><q>Tak by mě zajímalo,</q> začal jsi, ale na druhé straně se ozval
            velitelův hlas:
            <.p><q>Neptej se na hlouposti, mladej,</q> odsekává z vysílačky,
            <q>a raději mi řekni, proč jsi takhle potichounku zmizel ze
            základny!</q> ',
        '<.p>Začal jsi mluvit, ale než jsi stihl dopovědět, už tě velitel
            přerušil:
            <.p><q>Nech těch blbech řečí a odpověz,</q> syknul z vysílačky. '
    ]
;

/* ------------------------------------------------------------------------ */
/*
 *   Nyní by se měl začít naopak ptát hráč a konfrontovat velitele a kapitána s
 *   různými věcmi, které zjistil. V tuto dobu by měl být už pořádně naštvaný a
 *   motivovaný k přejití z defenzivy do protiútoku. Chceme, aby se neudržel a
 *   nakonec se zeptal na něco z trezoru. Jakmile se ukáže, že se hrabal v
 *   trezoru, pokusí se ho velitel podplatit a odradit od toho, aby dojel na
 *   konec cesty.
 */
+ ConvNode 'questions'

    /* V tomto konverzačním uzlu nenabízíme jiná témata. */
    limitSuggestions = true

    isSticky = true

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
        { return nil; }

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku.
     */
    npcContinueList: ShuffledEventList
    {
        [
            'Do tvého vědomí se prodírá hlas velitele opakovaně mluvící o
                seznamu tvých prohřešků. Asi je vhodný čas ujmout se slova. ',
            'Ten velitelův hlas je jízlivý a neodbytný: <q>Tak co s tebou,
                mladej?</q> Nejvyšší čas se přestat nechat buzerovat! proběhlo
                ti hlavou, pánové mi budou muset vysvětlit spoustu věcí! ',
            '<q>Haló? Vnímáš mě vůbec?</q> ptá se velitelův hlas a ty znovu
                přemítáš, jak udržet situaci, aby se nedostala zcela pod
                kontrolu těch dvou. Budeš do toho muset ostře říznout. '
        ]
        eventPercent = 67
    }

    commonFollowup()
    {
        "<.p><q>Henry, slyšíš to?</q> hlas z vysílačky zní tlumeně, jakoby hovor
            probíhal někde stranou: <q>Henry! Ten hajzl se dostal do trezoru.
            Proč jsi ho sem tahal? To sis nemohl vybrat nějakého lepšího
            blbečka? To ti nestačila jedna zkušenost?</q>

            <.p><q>POSLOUCHEJ, MLADEJ!</q> auu, tohle bylo na plné pecky, že bys
            uskočil, kdyby bylo kam: <q>Vždyť se můžeme domluvit. Viděl jsi
            diamanty a je jich dost i pro tebe a bude jich ještě víc.</q> hlas
            velitelé je tvrdší než ty diamanty, ale přitom zářivě jasně před
            tebou vykresluje budoucnost: <q>Jsi mladej, život máš před sebou a
            pokud se domluvíme, budeš zajištěnej na vždycky. Otevře se ti celý
            vesmír, žádné kodrcání na dopravních transportérech. Řekni ano a
            otoč to zpátky.</q><.convnode bribe><.reveal hint-bribe> ";
    }
;

++ HelloTopic "Už spolu komunikujete. ";

++ ByeTopic, StopEventList
    [
        'Řekl jsi do vysílačky: <q>Končím!</q> ale se zlou ses potázal,
            velitelův hlas tě poučil o zásadách slušného chování tak důrazně, že
            lituješ, žes vůbec otvíral ústa. ',
        'Pokusil ses hovor ukončit, ale jednak se ti vůbec nepodařilo skočit
            veliteli do řeči a jednak vysílačka nejde vypnout. '
    ]
;

/*
 *   Nabízí se spousta všetečných otázek. Vetšina z nich jsou ale jen dohady,
 *   které dokáží velitel s kapitánem zpochybnit nebo vysvětlit výmluvou. My ale
 *   chceme hráče dohnat k otázce na něco z trezoru, aby prozradil, že se v něm
 *   hrabal. Proto tyto s neuspokojivými odpovědmi doprovázíme povzbuzením z
 *   commander.disapointingAnswer, které směřují hráče k zeptání se na něco
 *   podstatnějšího.
 */
++ AskTellTopic, SuggestedAskTopic @suspiciousContainer
    "<.p><q>Tak do toho,</q> pomyslel sis a nahlas jsi řekl: <q>Přivezli jsme s
        sebou bednu, která nebyla v papírech a honem se uklízela pryč. Co v ní
        bylo?</q>

        <.p><q>Bedna?</q> jakoby ti na druhé straně špatně slyšeli, <q>ty myslíš
        tu bednu, co zůstala venku? To ses měl zeptat rovnou.</q> Úplně vidíš
        velitele, jak se usmívá a shovívavě ti vysvětluje: <q>Jako bys to
        neznal, pár drobností na přilepšení zdejšího života.</q>
        <<commander.disapointingAnswer>>"

    /*
     *   Jestliže nastavíme téma jako doporučené (SuggestedAskTopic), aby se na
     *   něj hráč zeptal, musíme napsat, jak bude téma označené v nabídce, tj.
     *   uvedeme název objektu ve čtvrtém pádě (zeptat se na koho/co). V případě
     *   SuggestedTellTopic pak mohl bys mu říci o kom/čem 6. pád.
     */
    name = 'podezřelou bednu'
;

++ AskTellTopic, SuggestedAskTopic @tIncident
    "<.p>Jo, tohle ti taky vrtá hlavou. Jak se mohl stroji, určenému k těžbě v
        extrémních podmínkách v nejtvrdších horninách, zničit na pár metrech
        razící štít? Ta oficiální zpráva přece nedala spát ani lidem ze
        společnosti. Takže do toho: <q>Co se vlastně stalo s tím rypadlem?</q>
        položil jsi dotaz.

        <.p><q>Hele, mladej, tohle překračuje tvé kompetence,</q> odsekl
        velitel, <q>ale pokud tě to tak zajímá, naše zpráva společnosti je
        dostatečně podrobná a běžně přístupná.</q>
        <<commander.disapointingAnswer>> "

    name = 'nehodu rypadla'
;

++ AskTellTopic, SuggestedAskTopic @sq71
    "<.p><q>Dobře,</q> prohlásil jsi směrem k mikrofonu, <q>tvrdíte, že SQ71 je
        zakázaná oblast, a přitom jste tam sami jeli, proč?</q>

        <.p><q>Chlapečku,</q> ozvala se jedovatá odpověď, <q>já jako velitel
        téhle základny ti vůbec nemusím skládat účty z toho, co dělám nebo kam
        jedu. Chápeš?</q><<commander.disapointingAnswer>> "

    name = 'zakázanou oblast'
;

++ AskTellTopic, SuggestedAskTopic @fixedComponents
    "<.p><q>Můžu se zeptat,</q> rozhodl ses položit otázku, <q>kde jste vzali ty
        náhradní součástky?

        <.p><q>Tss,</q> znechucení z velitelova hlasu přímo odkapává, <q>to ses
        mohl zeptat rovnou a nedělat kolem takový tyátr. Vytáhli jsme je z
        vyřazených rypadel CMX12.</q><<commander.disapointingAnswer>> "

    name = 'opravené součástky'
    isActive = !tCmx12.isKnown
;

++ AskTellTopic, SuggestedAskTopic [fixedComponents, tCmx12]
    "<.p><q>Můžu se zeptat,</q> rozhodl ses položit otázku, <q>kde jste opravdu
        vzali ty náhradní součástky?</q>

        <.p>Podle hlasu vypadá velitel fakt dost namíchnutě: <q>Čemu jsi
        nerozumněl na větě, že jsme je vytáhli z vyřazených rypadel CMX12?</q>

        <.p>Jaká výhoda, že nestojíš přímo vedle něj, jaká škoda, že neuvidíš,
        jak se tváří: <q>Jenže, veliteli, na základně žádné CMX nejsou a nikdy
        ani nebyly.</q>

        <.p><q>Hele, mladej,</q> odsekne velitel, <q>nejseš nějak moc chytrej?
        Co ty můžeš vědět, co na svém asteroidu mám nebo nemám.</q>
        <<commander.disapointingAnswer>> "

    name = 'opravené součástky'
    isActive = gRevealed('cmx12-lie')
;

++ AskTellTopic, SuggestedAskTopic @tMining
    "<.p><q>Veliteli,</q> snažíš se, aby tvůj hlas zněl trošku znuděně, <q>proč
        došlo k přesunu těžby?</q>

        <.p><q>Ty snad vůbec nečteš,</q> podivuje se velitel, <q>říká ti něco
        vytěženost a bohatší ložiska?</q>

        <.p><q>Právě, že říká,</q> oponuješ, <q>viděl jsem přehledy poklesu
        těžby po změně dolu.</q>

        <.p><q>Tak to asi ne ty správné, pojeď zpátky a ukážu ti je,</q> přímo
        se vlichocuje velitelův hlas.<<commander.disapointingAnswer>> "

    name = 'přesun těžby'
;

++ AskTellTopic @tChriss
    "<.p>Tuhle otázku prostě musíš položit: <q>Veliteli, co se vlastně stalo
        mému předchůdci, Chrissovi?</q>

        <.p><q>Tss,</q> pohrdavě odfrknul velitel, <q>náhoda a je z ní nehoda.
        Ve vesmíru se to může stát kdykoliv a kdekoliv.</q>
        <<commander.disapointingAnswer>> "

    name = 'na Chrisse'
;

++ AskTellTopic [tresorBills, tresorLetter]
    "<.p><q>Veliteli,</q> rozhodl ses pro čelní útok, <q>řekněte mi, k čemu tady
        potřebujete tak specializované a tak drahé laboratorní vybavení.
        Společnost určitě bude zajímat, jak vědychtivé lidi má na
        asteroidech.</q><<location.commonFollowup>> "
;

++ AskTellTopic @tresorPapers
    "<.p><q>Veliteli,</q> rozhodl ses pro čelní útok, <q>řekněte mi, s kým a na
        co chcete spolu s mým kapitánem uzavřít tu smlouvu? Společnost neví, že
        plánujete na jejím asteroidu milionové kšefty, že?</q>
        <<location.commonFollowup>> "
;

++ AskTellTopic [tresorDiamonds, tresor]
    "<.p><q>Veliteli,</q> rozhodl ses pro čelní útok, <q>řekněte mi, kde jste
        přišel k těm diamantům? Z platu velitele základny jste je nenakoupil a
        tady na asteroidu nic takového není. Společnost bude zajímat, jak se
        daří lidem na asteroidech.</q><<location.commonFollowup>> "
;

++ DefaultAskTopic, ShuffledEventList
    [
        '<.p><q>Pane,</q> začal jsi svůj dotaz, ale velitelova odpověď tě tvrdě
            odpálkovala:
            <.p><q>Nezdá se ti, že tohle je úplně mimo mísu?</q> ',
        '<.p><q>Veliteli, poslyšte,</q> vznesl jsi dotaz k veliteli, ale ten jen
            pohrdavě odfrknul:
            <.p><q>K čemu o takových hloupostech potřebuješ mluvit?</q> ',
        '<.p>Zeptal ses, ale odpovědí byl jen hustý stěr:
            <.p><q>O tom se s tebou vůbec nebudu bavit,</q> řekl velitel,
            <q>máme tu jiné problémy.</q> '
    ]
;

++ DefaultAnyTopic, ShuffledEventList
    "<.p><q>Pane,</q> začal jsi, ale přerušilo tě zavrčení:
        <.p><q>Vrať se k tématu!</q> "
;

/* ------------------------------------------------------------------------ */
/*
 *   Jako poslední možnost se snaží velitel hráče uplatit. Ve skutečnosti to
 *   nemyslí tak upřímně, ale to nevíme. Každopádně to je poslední pokus hráče
 *   odradit od cíle své cesty.
 */
+ ConvNode 'bribe'

    /* V tomto konverzačním uzlu nenabízíme jiná témata. */
    limitSuggestions = true

    isSticky = true

    /* Nedovolíme hráči utéci z otázky tím, že by zakončil konverzaci. */
    canEndConversation(actor, reason)
        { return nil; }

    /*
     *   Toto jsou hlášky, které se zobrazí, pokud hráč dělá něco jiného, než že
     *   by odpověděl na položenou otázku.
     */
    npcContinueList: ShuffledEventList
    {
        [
            '<q>Tobiáši,</q> ozvalo se z vysílačky, <q>vždyť nemáš nad čím
                váhat. Pojeď zpátky, oslavíme naše nové spojenectví.</q> ',
            '<q>Tak nad čím váháš?</q> velitelův hlas se chvěje netrpělivostí,
                možná z něm slyšíš i nejistotu, jaká bude tvoje odpověď. '
        ]
        eventPercent = 67
    }
;

++ HelloTopic "Už spolu komunikujete. ";
++ ByeTopic "Teď nemůžeš hovor ukončit, velitel trvá na odpovědi. ";

++ YesTopic, SuggestedYesTopic
    topicResponse()
    {
        "<.p>Ta vidina celoživotní jistoty je lákavá, strávil jsi příliš roků
            uskrovňováním po internátech a teď bys byl konečně bez starostí o
            centíky: <q>A jakou mám jistotu, že mě nepodrazíte?</q> zeptal ses
            do vysílačky.

            <.p><q>Tady Henry Scott,</q> ozval se z vysílačky druhý hlas, <q>máš
            mé slovo kapitána. Pojeď zpátky.</q>

            <.p>Otočil jsi vozítko a rozjel se zpět na základnu. Tam tě uvítali
            oba velitelé z úsměvem a před odletem ti věnovali dva krásné
            diamanty -- jeden jako druhý.

            <.p>Cestu zpátky jsi trávil v příjemném snění, jak si okamžitě
            vylepšíš život. O to tvrdší probuzení bylo na Zemi, kde tě hned po
            přistání zatkla policie s obviněním z pašování drog. V tvých
            zavazadlech byl nalezen balíček drog a diamanty byly zabaveny jako
            důkaz platby za pašování. U soudu vypovídal kapitán proti tobě, že
            tvé chování při letu budilo podezření ovlivnění drogami. Cos mohl
            říci? Tvému obvinění kapitána Scotta a velitele základny Petronova
            se všichni jen smáli. Takže teď čekáš na rozsudek, jestli vůbec
            někdy z kriminálu vylezeš. Drogy ve vesmíru jsou trestány víc než
            tvrdě. ";

#ifdef TADS_INCLUDE_NET
//        local data =
//        [
//            'sid' -> webSession.storageSID,
//            'event' -> 'lost',
//            'points' -> 0,
//            'desc' -> '',
//            'total' -> libScore.totalScore
//        ];

//        sendNetRequest(0, 'http://technoplaneta.cz/t3submit', 'POST', 0, nil,
//            data);
#endif

        finishGameMsg('PROHRÁL JSI', [ finishOptionUndo, finishOptionCredits,
            finishOptionFullScore ]);
    }
;

++ NoTopic, SuggestedNoTopic
    topicResponse()
    {
        "<.p>Vidíš v duchu to blýskání kamínků, poprvé a možná naposled v životě
            jsi tam u trezoru v ruce držel opravdovský diamant, ale víš, že
            pokud bys kývnul na jejich lákání, v životě by ses nemohl ani
            podívat do zrcadla sobě do očí. Nadechl ses, naklonil k vysílačce a
            zřetelně jsi vyslovil: <q>NE!</q>

            <.p>A dřív, než se ozval někdo z druhé strany, jsi pokračoval:
            <q>Jedete v nějaké křivárně, jinak byste se takhle nevnucovali, a já
            příjdu na to, co to je.</q>

            <.p>Ticho, jak by vysílačka nad tvou odvahou (nebo drzostí) úžasem
            oněměla. Až po chvíli se ozvalo ledové: <q>Tvoje rozhodnutí, skončíš
            jako ten šťoura před tebou!</q>

            <.p>Cvak. A ticho.

            <.p>Takové ticho, že slyšíš rámus ve své hlavě: <q>Tím šťourou přede
            mnou určitě mysleli Chrisse, chudák, takže někdo té náhodě pomohl.
            To tedy nebyla nehoda, byla to sprostá vražda. Kurňa, do čeho jsem
            to šláp,</q> pomyslel sis.

            <.p>Konečně se na navigaci ukazatel polohy doplazil až na souřadnice
            označující cíl. Ve světlech reflektorů se pomalu ukázala rozšiřující
            se šachta.

            <.p>Posunul jsi řídicí páku skoro do neutrální polohy a vozítko
            začalo zpomalovat. Á, tamhle bude dobré místo k zastavení.

            <.p>Stojíš, ale na navigaci se něco pohybuje -- bod právě se
            vzdalující od základny a vypadá, že najíždí na tvou stopu.
            <.convnode nil> ";

        commander.setCurState(commanderChasing);
        captain.setCurState(captainChasing);

        gReveal('rover-destination');

        clock.eventReached();
    }
    clock: ClockEvent { eventTime = [2, 14, 58] }
;

++ DefaultAnyTopic, StopEventList
    [
        '<.p><q>No, já bych,</q> začal jsi, ale přerušil tě velitelův hlas
            podbarvený nervozitou:
            <.p><q>Počkej, tohle teď řešit nebudeme. Prostě kývni na naši
            nabídku a pojeď zpátky.</q> ',
        '<.p>Pokoušíš se vyhnout jasné odpovědi, ale hlas z vysílačky tě
            přerušuje:
            <.p><q>Hergot, kluku,</q> velitel zřejmě balancuje na pokraji
            hysterického záchvatu, <q>nemluv o hovadinách a otoč to na základnu,
            ať si plácneme.</q> '
    ]
;

/* ------------------------------------------------------------------------ */
/*
 *   KAPITÁN S VELITELEM PRONÁSLEDUJÍ HRÁČE
 */
+ commanderChasing: HermitActorState
    stateDesc = "Sedí za kniplem a má odhodlaný výraz. "

    /*
     *   Nechame umyslne prazdne a pritom zadane, aby se negeneroval implicitni
     *   group listing.
     */
    specialDesc = ""
    noResponse = "Velitel se ve vysílačce nehlásí, spojení je přerušené. "
;
