#charset "utf-8"

/*
 *   Základna na asteroidu
 *
 *   Původní česká textová hra, která vznikla jako speciál pro dětskou šifrovací
 *   hru Technoplaneta.
 *
 *   Copyright © 2015, Tomáš Bláha, Pavel Čejka, Milan Vančura a Alena Vávrová.
 *   Všechna práva vyhrazena.
 *
 *   Mechanismus zobrazování tipů pochází z hry Mrs. Pepper's Nasty Secret od
 *   Jima Aikina a Erica Eva. Použito s laskavým svolením.
 */

#include <adv3.h>
#include <cs_cz.h>

#define gActionWas(action)\
    (me.lastAction != nil && me.lastAction.actionOfKind(action##Action))
#define gLastDobj (me.lastAction.getDobj())
#define gLastIobj (me.lastAction.getIobj())
#define gLastTopic (me.lastAction.getTopic())
#define gLastTopicText (gLastTopic.getTopicText.toLower())

/*
 *   FRAMEWORK FOR THE EXTRA HINTS MECHANISM
 *
 *   The ExtraHint class provides a framework for providing players
 *   (especially novice players) with extra hints, nudges, and tips during
 *   the course of play. When it is activated (either at player request, or
 *   because the player appears to be having trouble), a hint or tip can be
 *   displayed at the end of the turn in response to what the player has
 *   just typed, or some condition that has just become true, or which
 *   became true a certain number of turns ago.
 *
 *   It is also possible to define conditions (typically the achievement of
 *   some goal) that render a particular ExtraHint otiose, so that it is not
 *   displayed if it is no longer needed.
 */
modify Achievement
    achieved = (scoreCount > 0)
;

modify Actor
    lastAction = nil
    actorAction()
    {
        if(!gAction.isImplicit)
            lastAction = gAction;
    }
;

/* Command for enabling or disabling extra hints */

VerbRule(ExtraHints)
    ('zapnout'->onOff | 'zapni'->onOff | ) ('tutoriál' | 'výuku' | 'výuka')
    :ExtraHintsAction
    verbPhrase = ('zapnout/zaturning extra hints ' + onOff)
;

DefineSystemAction(ExtraHints)
    execSystemAction()
    {
        if(onOff == nil)
        {
            local cmd = onOrOff(!extraHintsActive).toUpper() + ' VÝUKU';
            "Výuka je momentálně <<onOrOff(extraHintsActive)>>. K
                <<onOrOff(!extraHintsActive)>> použij příkaz <<aHref(cmd,
                cmd, onOrOff(!extraHintsActive)) + ' výuku' >>. ";
            return;

        }
        onOff = onOff.toLower();
        if(onOff == 'vypnout')
            extraHintManager.stopDaemon();
        else
            extraHintManager.startDaemon();

        "Hotovo, výuka je teď <<onOff>>. ";
    }
    extraHintsActive = (extraHintManager.activated)
    onOrOff(stat) { return stat ? 'zapnutá' : 'vypnutá'; }
;

/*
 *   The ExtraHint class. We simply need to define objects of this class to
 *   encapsulate the extra hints we want displayed.
 */


class ExtraHint: object
    location = extraHintManager
    closeWhen()
    {
        if(hintDone || (achieveObj && achieveObj.achievement.achieved))
            return true;

        return checkClose;
    }

    /* The object whose achievement we monitor */
    achieveObj = nil

    /* The condition that needs to be true for us to offer this hint */
    openWhen = nil

    /*
     *   The number of turns between openWhen becoming true and this hint
     *   being displayed.
     */
    hintDelay = 0

    /*  Alternative condition for closing this ExtraHint */
    checkClose = nil


    /*
     *   If the closeWhen condition is true we remove this ExtraHint from the
     *   list of potentially active ExtraHints and return nil to tell the
     *   caller that this ExtraHint was not displayed. Otherwise we check if
     *   this ExtraHint (a) meets its openWhen conditions to be displayed
     *   and (b) is due to be displayed because hintDelay turns have passed
     *   since it was open. If both conditions are met we show our text and
     *   return true to tell our caller that an ExtraHint has been displayed
     *   (important since we display at most one ExtraHint per turn).
     *   Otherwise we return nil.
     */
    doHint()
    {
        if(closeWhen)
        {
            extraHintManager.removeFromContents(self);
            return nil;
        }

        if(openWhen)
        {
            if(openedWhen == nil)
                openedWhen = libGlobal.totalTurns;

            if(libGlobal.totalTurns < (openedWhen + hintDelay))
               return nil;

            showHint();
            return true;
        }
        return nil;
    }

    /* The text to display in relation to this ExtraHint */
    hintText = ""

    showHint()
    {
        "<.p><font color=\"#808080\">";
        hintText();
        "</font><.p>";
        hintDone = true;
    }

    hintDone = nil

    openedWhen = nil
;

/*
 *   A Template to facilitate the definition of ExtraHints. We can define it
 *   here and not in a header file since ExtraHints are only defined in this
 *   source file. */

ExtraHint template ->achieveObj? +hintDelay? "hintText";


/*
 *   The extraHintManager carries out the work of starting and stopping the
 *   display of ExtraHints, and determing which, if any, ExtraHint is ready
 *   to be displayed.
 */
extraHintManager: PreinitObject
    startDaemon()  { daemonID = new Daemon(self, &hintDaemon, 1); }

    stopDaemon()
    {
        if(daemonID)
        {
            daemonID.removeEvent();
            daemonID = nil;
        }
    }

    daemonID = nil
    activated = (daemonID != nil)

    /*
     *   When we're active we run through every ExtraHint object in our
     *   contents and run its doHint() method till either we run out of
     *   ExtraHint objects or we find one that displays an extra hint, and
     *   so returns true.
     */
    hintDaemon()
    {
        /*
         *   Z ne úplně pochopitelného důvodu když běží démon poprvé, tak vše
         *   provede tvrdošíjně před vypsáním intra a rozhlédnutím v místnosti.
         *   Abychom to obešli, úplně první vyvolání zcela ignorujeme a místo
         *   toho si démona zavoláme sami ve výpisu místnosti.
         */
        if(firstRun)
        {
            firstRun = nil;
            return;
        }
        foreach(local obj in contents)
        {
            if(obj.doHint())
            {
                if(firstHint)
                {
                    //extraHintMsg;
                    firstHint = nil;
                }
                break;
            }
        }
    }

    firstHint = true
    firstRun = true

    contents = []

    addToContents(obj)   {  contents += obj; }

    removeFromContents(obj) {  contents -= obj; }

    /* Build a list of all ExtraHints in our contents property at PreInit */
    execute() { forEachInstance(ExtraHint, {x: addToContents(x) });  }

    explainExtraHints()
    {
        "If you\'re new to Interactive Fiction and would like to read a few
        extra hints and tips that will pop up here and there as you explore the
        story, type <<cmdStr('ON')>>. If you decide you don\'t want any more of
        these bonus tips, simply type <<cmdStr('OFF')>>. ";
    }
    cmdStr(stat)
    {
        "<<aHref('EXTRA ' + stat, 'EXTRA ' + stat, 'Turning extra hints ' +
                stat.toLower)>>";
    }
    extraHintMsg = "<.p><font color=\"#808080\">(Budeš-li chtít vypnout tyto
        bonusové tipy, zadej příkaz <<aHref('EXTRA OFF', 'EXTRA OFF', 'Vypnout
        bonusové tipy')>>)</font>";
;

/*
 *   A series of ExtraHints can now follow.
 *
 *   The first element in the template is the object whose achievement
 *   property we check to see if this ExtraHint has become redundant. For
 *   example, if the player has already taken the flyer, flyer.achievement
 *   has been achieved, and so we don't need this first ExtraHint.
 *
 *   The second element in the template (e.g. +1) is the hintDelay; this is
 *   number of turns that should elapse between the openWhen condition (see
 *   below) becoming true and the hint being offered. This optionally allows
 *   the player a few turns to work on the solution before being offered a
 *   gratuitous hint.
 *
 *   The third element in the template is simply the text of the extra hint
 *   to display.
 *
 *   The openWhen property holds the condition that must be true before this
 *   ExtraHint is displayed. In the case of the first ExtraHint, the
 *   condition is that the player char has visited the location that
 *   contains the telephone pole.
 */

/* ------------------------------------------------------------------------ */
/*
 *   ZÁKLADY
 */
+ ExtraHint
    "<b>Tip:</b> Příběh se ovládá psaním jednoduchých příkazů za šipečku níže.
        První, co by ses měl naučit, je prozkoumat všechno kolem sebe. Příkaz
        PROZKOUMEJ <i>NĚCO</i> ti řekne více o objektech, které byly v popisu
        lokace zmíněny. Tady ve své kóji můžeš zkoumat přihrádky, síťovou kapsu,
        oblečení atp., ale asi jako úplně první se budeš chtít dozvědět více o
        své postavě, takže začni napsáním PROZKOUMEJ SEBE a zmáčkni Enter. "

    /*
     *   Podmínka určuje, kdy máme zobrazit tip hráči. Tento tip zobrazíme
     *   okamžitě na začátku hry. Každý tip se zobrazí jen jednou, jakmile je
     *   použit, HintManager ho vyřadí.
     */
    openWhen = true
;

+ ExtraHint
    "<b>Tip:</b> Vida, teď už víš, že jsi na cestě ve vesmírné lodi. Jak by ses
        ještě mohl dozvědět, kam a proč vlastně letíš? "

    /*
     *   Tip zobrazíme, jakmile hráč prozkoumá sám sebe. Pokud by však
     *   prozkoumal rozkazy nebo už nastala porucha lodi, je tip už nepotřebný a
     *   zahodíme ho.
     */
    openWhen = me.described
    checkClose = orders.described || gRevealed('ship-malfunction')
;

+ ExtraHint
    "<b>Tip:</b> Příkaz PROZKOUMEJ ti o předmětu prozradí jen základní
        informace, které jsou vidět na první pohled. Postava s předmětem při
        zkoumání nijak nemanipuluje, jen zaměří svou pozornost a řekne, co vidí.
        Tenhle předmět ale můžeš také číst, vyzkoušej příkaz PŘEČTI. "

    openWhen = manual.described || orders.described
;

+ ExtraHint
    "<b>Tip:</b> Oblečení a další věci, které můžeš mít na sobě, si navlékneš
        příkazy OBLÉKNI/OBUJ/NASAĎ SI <i>NĚCO</i>. Jsou to synonyma, je jedno,
        který příkaz použiješ. Opakem jsou pak příkazy SVLÉKNI/ZUJ/SUNDEJ, popř.
        můžeš rovnou oblečení v lokaci ODLOŽIT. "

    openWhen = overal.described || tshirt.described || shoes.described
    checkClose = gRevealed('clothes-worn')
;

+ ExtraHint
    "<b>Tip:</b> Přihrádky, skříňky a podobné objekty můžeš otevírat příkazem
        OTEVŘI <i>NĚCO</i>, a tak budeš moci nahlédnout dovnitř. Samozřejmě co
        otevřeš, můžeš také zavřít. Příkazy OTEVŘI a ZAVŘI můžeš také použít na
        dveře, které ve hře potkáš, i když to většinou není nutné, protože se
        otevřou samy při tvém průchodu. "

    openWhen = ycLeftCompartment.described || ycRightCompartment.described

    /*
     *   Kontrola nepotřebných tipů se provádí v každém tahu, takže jakmile hráč
     *   poprvé otevře některou z přihádek, tip se zahodí.
     */
    checkClose = ycLeftCompartment.isOpen() || ycRightCompartment.isOpen()
;

+ ExtraHint
    "<b>Tip:</b> Gratuluji, našel jsi tablet. Mnoho předmětů, které ve hře
        najdeš, si budeš moci vzít. Napiš příkaz VEM TABLET a tím se zařadí do
        tvého inventáře, tedy předmětů, které postava nosí s sebou. Kdykoliv se
        na nesené předměty můžeš zeptat zadáním příkazu INVENTÁŘ (můžeš zkrátit
        na pouhé I). "

    openWhen = ycRightCompartment.isOpen()
;

+ ExtraHint +1
    "<b>Tip:</b> Velmi důležitou věc, kterou se teď ještě musíš naučit, je
        umisťování předmětů DO či NA jiné. Vyzkoušej si např. příkaz DEJ ROZKAZY
        DO PRAVÉ PŘIHRÁDKY. Někdy můžeš vložit předmět i tam, kam bys zprvu
        nečekal. Schválně, kam kromě schránek a síťové kapsy se dá tady v
        místnosti něco strčit? "

    openWhen = ycRightCompartment.isOpen()
;

+ ExtraHint
    "<b>Tip:</b> Nezapomeň na prozkoumání svých rozkazů, které tu jsou položené.
        Dozvíš se z nich něco o své cestě vesmírem. "

    /*
     *   Metoda isWornBy informuje o tom, zda mám oblečení na sobě, me.location
     *   je lokace, ve které se aktuálně nachází hráč a vlastnost moved u
     *   objektu je true, jakmile ho hráč alespoň jednou sebral.
     */
    openWhen = tshirt.isWornBy(me) && overal.isWornBy(me) && shoes.isWornBy(me)
    checkClose = me.location != yourCabin || orders.described || orders.moved
;

+ ExtraHint
    "<b>Tip:</b> Teď, když už jsi ustrojený a seznámil se s věcmi ve své kóji,
        můžeš se vydat na průzkum vesmírné lodi. V záhlaví pod nadpisem
        místnosti uvidíš směry, kterými se můžeš vydat. Tady na palubě kosmické
        lodi to jsou směry na pravobok, levobok, příď a záď. Napiš příkaz JDI
        NA PRAVOBOK. "

    openWhen = orders.described && tshirt.isWornBy(me) && overal.isWornBy(me)
        && shoes.isWornBy(me)
    checkClose = me.location != yourCabin
;

+ ExtraHint
    "<b>Tip:</b> Výborně, právě ses přesunul do další místnosti. Aby sis
        ušetřil psaní, můžeš příkazy přesunů zkrátit na LB a PB pro pohyb směrem
        na levobok a pravobok, PD a ZD pro pohyb na příď a záď, N a D pro nahoru
        a dolů. Někdy lze jít také DOVNITŘ a VEN. Mimo vesmírnou loď se pak
        chodí ve směru světových stran, které se zkracují na jedno písmenko,
        tedy S, J, V, Z.
        <.p>Ale než se vydáš dál, nezapomeň tu nejprve všechno prozkoumat, což
        můžeš zkrátit na pouhé P <i>PŘEDMĚT</i>. "

    /* Vlastnost visited místnosti je true, jakmile do ní hráč vešel. */
    openWhen = livingRoom.visited
;

+ ExtraHint
    "<b>Tip:</b> Většinou nezáleží na tom, kde přesně v místnosti stojíš.
        Postava automaticky vidí a dosáhne na všechny předměty. Narazíš ale na
        objekty, které přímo vyzývají k tomu, aby sis na ně stoupl, sedl či
        lehl, protože tím něčeho dosáhneš. Vyzkoušej příkaz STOUPNI SI NA PÁS,
        budeš si moci na něm zaběhat. "

    openWhen = treadmill.described

    /* Metoda isIn testuje, tedy zda hráč stoupl na pás. */
    checkClose = me.isIn(treadmill)
;

+ ExtraHint
    "<b>Tip:</b> Když prozkoumáš některé složitější objekty, příběh tě často
        upozorní, že se tyto objekty skládají z více částí. Prozkoumání pásu
        říká, že pás má přítlačné popruhy a pultík s ovládáním. Chceš si
        zaběhat? Tak prozkoumej ještě pultík. A i ten pultík má na sobě pár
        samostatně prozkoumatelných detailů. "

    openWhen = me.isIn(treadmill)
    checkClose = treadmillPanel.described
;

+ ExtraHint
    "<b>Tip:</b> Regulátor je otočný, a tak můžeš použít příkaz OTOČ REGULÁTOR
        NA <i>HODNOTU</i> nebo NASTAV REGULÁTOR NA <i>HODNOTU</i>. Hodnotou jsou
        zmíněná čísla od 0 do 3. Někdy se můžeš setkat s otočnými předměty,
        které však nemají dané polohy natočení. Pak stačí říci OTOČ
        <i>PŘEDMĚT</i>. "

    openWhen = treadmillDial.described || treadmillPanel.described
    checkClose = gRevealed('treadmill-operational')
;

+ ExtraHint
    "<b>Tip:</b> Chceš-li, aby ve hře uběhlo trochu času, můžeš zadat příkaz
        ČEKEJ. A nebo můžeš čas strávit zkoumáním různých věcí. Všiml sis
        například, že pás má počítadlo uběhnuté vzdálenosti? "

    openWhen = gRevealed('treadmill-run')
    checkClose = gRevealed('treadmill-leaved')
;

+ ExtraHint +1
    "<b>Tip:</b> Jakmile budeš mít dost běhání, můžeš pás zastavit a pak ho
        opustit příkazem SLEZ Z PÁSU, nebo prostě jen SLEZ. Podobně když na
        něčem sedíš nebo ležíš, tak můžeš VSTÁT, a vždy funguje i jednoduché
        VEN. "

    openWhen = gRevealed('treadmill-run')
    checkClose = gRevealed('treadmill-leaved')
;

+ ExtraHint
    "<b>Tip:</b> Ve textových hrách často narazíš na různé přístroje, které se
        dají ovládat. Obvykle mívají různá tlačítka, která se dají STISKNOUT, či
        páky, za které se dá ZATÁHNOUT či jimi POHNOUT. "

    openWhen = appHome.described || tablet.described || tabletButton.described
    checkClose = gRevealed('ship-landed')
;

/*
 *   Tento tip ma mit nizsi prioritu, nez konkretni tipy v prvni mistnosti,
 *   proto je umisten az po nich. ExtraHintManager je bere v pořadí, v jakém
 *   jsou ve zdrojáku.
 */
+ ExtraHint +3
    "<b>Tip:</b> Kdyby sis potřeboval připomenout, co se kolem tebe nachází, tak
        vždy můžeš zadat příkaz ROZHLÉDNI SE (který se dá zkrátit na R) a příběh
        znovu popíše tvé okolí. "

    openWhen = true
;

+ ExtraHint
    "<b>Tip:</b> Sice se tím nemusíš řídit, ale když tě nějaká postava vyzývá,
        abys šel s ní, můžeš zadat příkaz NÁSLEDUJ. "

    openWhen = gRevealed('captain-follow')
;

+ ExtraHint
    "<b>Tip:</b> S kniplem a jinými pákami se dá hýbat. Můžeš např. zadat příkaz
        POHNI KNIPLEM. "

    openWhen = gRevealed('joystick')
    checkClose = gRevealed('ship-landed')
;

+ ExtraHint
    "<b>Tip:</b> Skrz průzor a různá jiná okna ve hře se můžeš podívat ven.
        Vyzkoušej příkaz KOUKNI PRŮZOREM. "

    openWhen = cWindow.described
;

+ ExtraHint
    "<b>Tip:</b> Ohřívač je plně funkční. Všimni si, jak hra popisuje, že má
        dvířka a zapínací knoflík. Naznačuje ti tím, že můžeš OTEVŘÍT či ZAVŘÍT
        DVÍŘKA, abys mohl DÁT <i>NĚCO</i> DO OHŘÍVAČE a pak STISKNOUT KNOFLÍK,
        čímž se spustí ohřev. A funguje také jednoduché ZAPNI OHŘÍVAČ. "

    openWhen = owen.described
    checkClose = gRevealed('ship-malfunction')
;

/* ------------------------------------------------------------------------ */
/*
 *   TIPY PRO ROZHOVORY
 */
+ ExtraHint
    "<b>Tip:</b> S kapitánem si můžeš kdykoliv promluvit. Stačí napsat příkaz
        MLUV S KAPITÁNEM. "

    openWhen = cockpit.visited
;

/* Tyto dva tipy maji vyssi prioritu, nez obecny tip ke zkracovani ZEP/ŘEK */
+ ExtraHint
    "<b>Tip:</b> Takovouto drobnou výmluvou se ti příběh snaží jemně naznačit,
        že postava o tématu nic neví. Obvykle je nejlepší mluvit o konkrétních
        objektech, které jsi ve hře viděl, ale není důvod se nezeptat ani na
        další témata, o kterých byla ve hře řeč. "

    openWhen = gRevealed('default-topic')
;

+ ExtraHint
    "<b>Tip:</b> S postavou se komunikuje převážně tak, že se ptáš nebo ji říkáš
        o objektech, které jsou ve hře zmíněné. Vyzkoušej např. ZEPTEJ SE NA
        CESTU. Některá důležitá témata ti hra napoví, ale ne všechna. "

    openWhen = captain.curState == captainTalking
    checkClose = gActionWas(AskAbout) || gActionWas(TellAbout)
;

+ ExtraHint
    "<b>Tip:</b> Příkaz <<if gActionWas(AskAbout)>>ZEPTEJ SE
        <<gLastDobj.nameKohoCeho.toUpper>> NA<<else>>ŘEKNI
        <<gLastDobj.nameKomuCemu.toUpper>> O<<end>> <<gLastTopicText.toUpper>>
        můžeš, jakmile je už konverzace navázaná, zkrátit na
        <<if gActionWas(AskAbout)>>ZEP<<else>>ŘEK<<end>>
        <<gLastTopicText.toUpper>>. Podobně můžeš zkrátit
        <<if gActionWas(AskAbout)>>ŘEKNI <<gLastDobj.nameKomuCemu.toUpper>>
        O <i>NĚČEM</i><<else>>ZEPTEJ SE <<gLastDobj.nameKohoCeho.toUpper>> NA
        <i>NĚCO</i><<end>> na pouhé <<if gActionWas(AskAbout)>>ŘEK
        <i>NĚČEM</i><<else>>ZEP <i>NĚCO</i><<end>>.\b
        Příkazem TÉMATA získáš seznam možností, o kterých by stálo za to si s
        postavou promluvit, ale to neznamená, že bys nemohl mluvit i o dalších
        věcech. "

    /*
     *   Makro gActionWas kontroluje třídu akce. Třída je v TADSu jen speciálně
     *   označený objekt, takže ji můžeme poslat jako parametr do volání.
     */
    openWhen = (gActionWas(AskAbout) || gActionWas(TellAbout))
        && gLastDobj != nil

    /*
     *   Tady trochu vybočujeme z běžných kolejí. Všimněte si, že kdekoliv je
     *   očekávána nějaká vlastnost, můžeme podstrčit metodu vracející hodnotu.
     *   Využíváme toho, abychom se od parseru dozvěděli, zda se hráč už naučil
     *   používat zkratky konverzačních příkazů.
     */
    checkClose()
    {
        if(me.lastAction == nil) return nil;
        local text = me.lastAction.getOrigText();
        return text.match(R'<nocase>^(zep|[řr]ek)<space>');
    }
;

+ ExtraHint
    "<b>Tip:</b> Kromě ZEPTEJ SE NA či ŘEKNI O ti příběh občas nabídne možnosti,
        jak v rozhovoru pokračovat, které se tomuto schématu vymykají. Platí jen
        v ten daný okamžik a zvolíš je napsáním příkazu ve formě, jakou ti
        příběh nabízí. Teď ti příběh nabízí, že můžeš napsat PŘEVEZMI SLUŽBU. "

    openWhen = gRevealed('special-topic')
    checkClose = gRevealed('me-driving')
;

+ ExtraHint
    "<b>Tip:</b> Pokud ti zvýšená spotřeba paliva vrtá hlavou, můžeš se na ni
        kapitána zeptat. Napiš příkaz ZEPTEJ SE NA SPOTŘEBU. Obecně platí, že se
        můžeš postav ptát na více věcí, než jen na ty, které ti příběh sám
        doporučí při zahájení rozhovoru nebo v reakci na příkaz TÉMATA. A někdy
        to mohou být otázky dost důležité! "

    openWhen = gRevealed('fuel-consumption')
    checkClose = !captain.isIn(cockpit)
;

+ ExtraHint
    "<b>Tip:</b> Ačkoliv v tipech ke hře či v návodu jsou všechny ukázkové
        příkazy, které bys mohl zadat, odlišeny od okolního textu pomocí velkých
        písmen, ty tak rozhodně psát nemusíš. "

    openWhen = gRevealed('hint-lowercase')
;

+ ExtraHint
    "<b>Tip:</b> Nezapomeň, že se obvykle můžeš ptát na mnohem více věcí, než
        které ti hra napovídá. "

    openWhen = gRevealed('hint-topics')
;

/* ------------------------------------------------------------------------ */
/*
 *   ŘEŠENÍ PUZZLŮ
 */

+ ExtraHint +20
    "<b>Tip:</b> Pokud ti vrtá hlavou, co se to stalo za nehodu, možná bys to
        mohl zjistit zkoumáním palubní desky. "

    openWhen = gRevealed('ship-malfunction')
    checkClose = manual.described || flightScreen.described
        || radarScreen.described || stateScreen.described
        || lrPanelLock.achievement.scoreCount > 0
;

+ ExtraHint +25
    "<b>Tip:</b> Pokud ti vrtá hlavou, kde by mohlo být servisní podpalubí, zkus
        zapojit trochu důvtipu. Cestu do podpalubí budeš muset objevit. "

    openWhen = gRevealed('hint-lower-deck')
    checkClose = lrPanelLock.achievement.scoreCount > 0
;

+ ExtraHint +35
    "<b>Tip:</b> Ještě stále jsi neobjevil servisní podpalubí? Nehledej pomoc v
        žádných plánech či manuálech, najdi ho fyzicky. Prostě si představ loď
        a zamysli se, kde by tak asi mělo prostorově být. Potom zkoumej tím
        směrem. "

    openWhen = gRevealed('hint-lower-deck')
    checkClose = lrPanelLock.achievement.scoreCount > 0
;

+ ExtraHint
    "<b>Tip:</b> Bohužel tu není žádné osvětlení a to z horní místnosti sotva
        osvítí vstup. Kdyby sis ale nějaký zdroj světla přinesl, umožnilo by ti
        to se zde porozhlédnout. "
    openWhen = gRevealed('ship-dark')
        && !(tablet.isLit && tablet.isIn(lowerDeck))
;

+ ExtraHint
    "<b>Tip:</b> Na terminálu můžeš psát příkazem ZADEJ <i>NĚJAKÝ TEXT</i> DO
        TERMINÁLU. "
    openWhen = gRevealed('computer-typing')
;

/* ------------------------------------------------------------------------ */

extraHintInit: InitObject
    execute()
    {
        extraHintManager.startDaemon();
        extraHintManager.hintDaemon();
    }
;

#ifdef __DEBUG

modify Goal
    openWhen = (inherited || me.showAllHints)
;

#endif
